<?php
/**
 * @package CleverOgre
 * @subpackage WP Job Manager - JobAdder Integration
 * @version 0.1.3
 * @since 0.1.1
 */

namespace JobAdder;

if (!defined('ABSPATH')) exit;

class JobListing {

    public static function init() {
        add_filter('job_manager_job_listing_data_fields', array(__CLASS__, 'job_listing_meta'), 10, 1);

        // API Resources
        add_filter('wpjm-jobadder_job_meta', array(__CLASS__, 'update_meta'), 10, 3);
        add_filter('wpjm-jobadder_job_data', array(__CLASS__, 'update_post_data'), 10, 3);
        add_action('wpjm-jobadder_job_updated', array(__CLASS__, 'update_post'), 10, 2);
    }

    public static function update_job($job) {
        if (!$job->Title) return false;

        $ad_id = (int)$job->attributes()->jid;

        // Get existing post data from JobAdder id if exists
        $post_id = \JobAdder\API\JobAds::getPostId($ad_id);

        // Check if sync is disabled
        if (!is_null($post_id) && is_int($post_id) && $post_id > 0 && !self::is_sync($post_id)) {
            return true;
        }

        // Get classification data
        $classification = array(
            'categories' => array(),
            'location' => '',
            'featured_image' => '',
            'salary' => '',
            'work_type' => '',
            'email_to' => '',
            'url' => '',
        );

        foreach ($job->Classifications->children() as $_classification) {
            switch ($_classification['name']) {
                case 'Category':
                    $classification['categories'][] = (string)$_classification;
                    break;
                case 'Sub Category':
                    $classification['categories'][] = (string)$_classification;
                    break;
                case 'Featured Image':
                    $classification['featured_image'] = (string)$_classification;
                    break;
                case 'Location':
                    $classification['location'] = (string)$_classification;
                    break;
                case 'Work Type':
                    $classification['work_type'] = (string)$_classification;
                    break;
            }
        }

        if (isset($job->Salary) && !empty((string)$job->Salary)) {
            $classification['salary'] = (string)$job->Salary;
        }
        if (empty($classification['salary'])) {
            $salaries = array();
            if (!empty((string)$job->Salary->MinValue)) $salaries[] = (string)$job->Salary->MinValue;
            if (!empty((string)$job->Salary->MaxValue)) $salaries[] = (string)$job->Salary->MaxValue;

            if (!empty($salaries)) $classification['salary'] = implode(' - ', $salaries);
        }

        if (isset($job->Apply->EmailTo) && !empty((string)$job->Apply->EmailTo)) {
            $classification['email_to'] = (string)$job->Apply->EmailTo;
        }

        if (isset($job->Apply->Url) && !empty((string)$job->Apply->Url)) {
            $classification['url'] = (string)$job->Apply->Url;
        }

        $classification = apply_filters('wpjm-jobadder_job_classification', $classification, $ad_id, $job);

        // Setup post data
        $post_data = array(
            'post_title' => (string)$job->Title,
            'post_name' => sanitize_title((string)$job->Title),
            'post_excerpt' => (string)$job->Summary,
            'post_content' => self::build_content($job),
            'guid' => (string)$ad_id,

            'post_status' => 'publish',
            'post_type' => 'job_listing',
            //'post_author' => 1,

            'ping_status' => 'closed',
            'comment_status' => 'closed',
        );

        // Get posted date
        if (isset($job->attributes()->datePosted) && !empty($job->attributes()->datePosted)) {
            $dt = \DateTime::createFromFormat('Y-m-d', $job->attributes()->datePosted);
            if ($dt !== false) {
                $post_data['post_date'] = $dt->format('Y-m-d H:i:s');
                if ($dt->getTimestamp() > time()) $post_data['post_status'] = 'future';
            }
        }

        // Setup post meta
        $post_meta = array(
            '_jobadder_enabled' => '1',
            '_jobadder_sync' => '1',
            '_adid' => (int)$ad_id,
            '_job_salary' => !empty($classification['salary']) ? (string)$classification['salary'] : '',
            '_job_salary_period' => (string)$job->Salary['period'],
            '_job_benefits' => (string)$job->Salary->Text,
            '_job_location' => (string)$classification['location'],
            '_job_image' => (string)$classification['featured_image'],
            '_application' => wp_http_validate_url($classification['url']) ? $classification['url'] : (is_email($classification['email_to']) ? $classification['email_to'] : get_option('admin_email')),
            '_apply_link' => (string)$classification['url'],
            '_jetpack_dont_email_post_to_subs' => '1',
            '_salary_min' => (string)$job->Salary->MinValue,
            '_salary_max' => (string)$job->Salary->MaxValue,
            //'_company_name' => '',
            //'_company_website' => '',
            //'_company_tagline' => '',
            //'_job_expires' => '',
        );

        // Apply filters to post data
        $post_data['meta_input'] = apply_filters('wpjm-jobadder_job_meta', $post_meta, $ad_id, $job); // Add meta input
        $post_data = apply_filters('wpjm-jobadder_job_data', $post_data, $ad_id, $job);

        // Insert or update job_listing post
        if (!is_null($post_id) && is_int($post_id) && $post_id > 0) {
            $post_data['ID'] = $post_id;
            $post_id = wp_update_post($post_data);
        } else {
            $post_id = wp_insert_post($post_data);
        }
        if (is_null($post_id) || $post_id <= 0) return false;

        // Set work type
        if (taxonomy_exists('job_listing_type') && !empty($classification['work_type'])) {
            $work_types = array();
            switch ($classification['work_type']) {
                case 'Permanent / Full Time':
                    $work_types['office'] = __('Office-Based', 'wpjm-jobadder');
                    $work_types['full-time'] = __('Full Time', 'wpjm-jobadder');
                    break;
                case 'Contract or Temp':
                    $work_types['contract'] = __('Contract', 'wpjm-jobadder');
                    break;
                case 'Part-time':
                    break;
                /* case 'Casual':
                    break; */
                default:
                    $work_types[sanitize_title($classification['work_type'])] = $classification['work_type'];
            }
            $work_types = apply_filters('wpjm-jobadder_job_listing_work_types', $work_types, $classification['work_type'], $job);

            $work_type_terms = array();
            foreach ($work_types as $slug => $label) {
                $term = get_term_by('slug', $slug, 'job_listing_type');
                if (!$term) $term = wp_insert_term($label, 'job_listing_type', array('slug' => $slug));

                if (is_array($term) && isset($term['term_id'])) {
                    $work_type_terms[] = $term['term_id'];
                } else if (is_object($term)) {
                    $work_type_terms[] = $term->term_id;
                }
            }
            $work_types = apply_filters('wpjm-jobadder_job_listing_work_type_terms', $work_type_terms, $job);

            wp_set_post_terms($post_id, (array)$work_type_terms, 'job_listing_type');
        }

        // Add Tags
        if (taxonomy_exists('job_listing_tag') && !empty($classification['categories'])) {
            $tag_ids = array();
            foreach ($classification['categories'] as $category) {
                $tag_term = get_term_by('name', $category, 'job_listing_tag');
                if (!$tag_term) $tag_term = wp_insert_term($category, 'job_listing_tag');

                if (is_array($tag_term) && isset($tag_term['term_id'])) {
                    $tag_ids[] = (int)$tag_term['term_id'];
                } else if (is_object($tag_term)) {
                    $tag_ids[] = (int)$tag_term->term_id;
                }
            }
            $tag_ids = apply_filters('wpjm-jobadder_job_listing_tag_terms', $tag_terms, $job);

            wp_set_post_terms($post_id, (array)$tag_ids, 'job_listing_tag', true);
        }

        do_action('wpjm-jobadder_job_updated', $post_id, $job);

        return true;
    }

    static function update_meta($post_meta, $adId, $job) {
        // Update Job Id
        $jobId = \JobAdder\API\JobAds::getJobIdByAd($adId, true);
        if (!$jobId) return $post_meta;
        $post_meta['_jobid'] = strval($jobId);

        // Update Company Id
        $companyId = \JobAdder\API\Companies::getCompanyIdByJob($jobId, $adId, true);
        if (!$companyId) return $post_meta;
        $post_meta['_companyid'] = strval($companyId);

        // Update Company
        $companyInfo = \JobAdder\API\Companies::getCompanyInfo($companyId);
        if (empty($companyInfo)) return $post_meta;

        if (isset($companyInfo['name'])) $post_meta['_company_name'] = (string)$companyInfo['name'];
        if (isset($companyInfo['summary']) && !empty($companyInfo['summary'])) $post_meta['_company_tagline'] = (string)$companyInfo['summary'];

        if (isset($companyInfo['primaryAddress'])) {
            // URL
            if (isset($companyInfo['primaryAddress']['url'])) $post_meta['_company_website'] = (string)$companyInfo['primaryAddress']['url'];

            // Address
            if (isset($companyInfo['primaryAddress']['city']) && isset($companyInfo['primaryAddress']['state'])) {
                $post_meta['_job_location'] = sprintf('%s, %s', (string)$companyInfo['primaryAddress']['city'], (string)$companyInfo['primaryAddress']['state']);
            } else if (isset($companyInfo['primaryAddress']['city'])) {
                $post_meta['_job_location'] = (string)$companyInfo['primaryAddress']['city'];
            } else if (isset($companyInfo['primaryAddress']['state'])) {
                $post_meta['_job_location'] = (string)$companyInfo['primaryAddress']['state'];
            }

            // Custom Location Settings
            $post_meta['custom_location'] = '1'; // Enable Custom Location
            if (isset($companyInfo['primaryAddress']['city'])) $post_meta['custom_location_city'] = (string)$companyInfo['primaryAddress']['city'];
            if (isset($companyInfo['primaryAddress']['state'])) $post_meta['custom_location_state'] = (string)$companyInfo['primaryAddress']['state'];
            if (isset($companyInfo['primaryAddress']['country'])) {
                switch (strtolower((string)$companyInfo['primaryAddress']['country'])) {
                    case 'canada':
                        $post_meta['custom_location_country'] = 'CA';
                        break;
                    default:
                        $post_meta['custom_location_country'] = 'US';
                        break;
                }
            }
        }

        // BUG: Company Social (data not available from API)

        return $post_meta;
    }

    static function update_post_data($post_data, $adId, $job) {
        // Update post date and expiration date
        $ad = \JobAdder\API\JobAds::getAd($adId);
        if (!$ad) return $post_data;

        if (isset($ad['postAt']) && !empty($ad['postAt'])) {
            $dt = \DateTime::createFromFormat(\DateTime::ISO8601, $ad['postAt']);
            if ($dt !== false) {
                $post_data['post_date'] = $dt->format('Y-m-d H:i:s');
                if ($dt->getTimestamp() > time()) $post_data['post_status'] = 'future';
            }
        }

        if (isset($ad['expireAt']) && !empty($ad['expireAt'])) {
            $dt = \DateTime::createFromFormat(\DateTime::ISO8601, $ad['expireAt']);
            if ($dt !== false) {
                $post_data['meta_input']['_job_expires'] = $dt->format('Y-m-d');
            }
        }

        return $post_data;
    }

    static function update_post($post_id, $job) {
        $jobId = get_post_meta($post_id, '_jobid', true);
        $companyId = get_post_meta($post_id, '_companyid', true);
        if (!is_numeric($jobId) || !is_numeric($companyId)) return;
        $jobId = intval($jobId);
        $companyId = intval($companyId);

        // Update Logo
        \JobAdder\API\Companies::getLogo($jobId, $companyId);

        // Force location update
        if (function_exists('af_wpjm_force_location_update')) {
            af_wpjm_force_location_update($post_id);
        }
    }

    private static function parse_terms($work_type) {
        return trim(explode('/', $work_type)[1]);
    }

    private static function build_content($job) {
        if (!$job->Title) return '';

        return apply_filters('wpjm-jobadder_content', $job->Summary . '<br />' . self::build_bullet_points($job->BulletPoints) . '<br />' . $job->Description);
    }

    private static function build_bullet_points($bullet_points) {
        if (is_object($bullet_points) && isset($bullet_points->BulletPoint)) {
            $bullet_points = (array)$bullet_points->BulletPoint;
        }
        if (!is_array($bullet_points) || empty($bullet_points)) return false;

        $list = '<ul>';
        foreach ($bullet_points as $point) {
            $list .= "<li>{$point}</li>";
        }
        $list .= '</ul>';

        return $list;
    }

    static function job_listing_meta($fields) {
        if (!self::is_enabled()) return $fields;

        $fields['_jobadder_sync'] = array(
            'label' => __('Automatically Sync with JobAdder Feed', 'wpjm-jobadder'),
            'type' => 'checkbox',
            'priority' => 0.90,
        );

        $fields['_adid'] = array(
            'label' => __('JobAdder Ad ID', 'wpjm-jobadder'),
            'type' => 'text',
            'placeholder' => '',
            'priority' => 0.91,
        );

        $fields['_jobid'] = array(
            'label' => __('JobAdder Job ID', 'wpjm-jobadder'),
            'type' => 'text',
            'placeholder' => '',
            'priority' => 0.92,
        );

        $fields['_companyid'] = array(
            'label' => __('JobAdder Company ID', 'wpjm-jobadder'),
            'type' => 'text',
            'placeholder' => '',
            'priority' => 0.93,
        );

        if (!array_key_exists('_job_salary', $fields)) {
            $fields['_job_salary'] = array(
                'label' => __('Job Salary', 'wpjm-jobadder'),
                'type' => 'text',
                'placeholder' => '',
                'priority' => 13,
            );
        }

        if (!array_key_exists('_job_benefits', $fields)) {
            $fields['_job_benefits'] = array(
                'label' => __('Job Benefits', 'wpjm-jobadder'),
                'type' => 'textarea',
                'placeholder' => '',
                'priority' => 14,
            );
        }

        return $fields;
    }

    public static function is_enabled($post_id = false) {
        if ($post_id == false) $post_id = get_the_ID();

        return get_post_meta($post_id, '_jobadder_enabled', true) === '1';
    }

    public static function is_sync($post_id = false) {
        if ($post_id == false) $post_id = get_the_ID();

        if (!self::is_enabled($post_id)) return false;

        return get_post_meta($post_id, '_jobadder_sync', true) === '1';
    }

    public static function update_jobs(array $ad_ids) {
        $ad_ids = array_map('intval', $ad_ids);
        update_option('_jobadder_ads', $ad_ids);

        self::remove_jobs();
    }

    private static function remove_jobs() {
        $current_ad_ids = get_option('_jobadder_ads');

        $removed_post_ids = get_posts(array(
            'post_type' => 'job_listing',
            'post_status' => array(
                'publish',
                'future',
            ),
            'meta_query' => array(
                'relation' => 'AND',
                array(
                    'key' => '_jobadder_enabled',
                    'value' => 'true',
                    'compare' => '=',
                ),
                array(
                    'key' => '_adid',
                    'value' => $current_ad_ids,
                    'compare' => 'NOT IN',
                ),
            ),
            'fields' => 'ids',
        ));

        if (empty($removed_post_ids)) return;

        foreach ($removed_post_ids as $post_id) {
            wp_trash_post($post_id);
        }
    }

}

\JobAdder\JobListing::init();
