<?php
/**
 * Plugin Name: LearnDash LMS - PhpWord Certificates
 * Plugin URI: https://plugins.cleverogre.com/plugin/learndash-certificates-phpword/
 * Description: Generate LearnDash pdf certificates based on Word document templates.
 * Version: 0.1.1
 * Requires at Least: 5.0
 * Requires PHP: 7.3
 * Author: CleverOgre
 * Author URI: https://cleverogre.com/
 * License: GPLv2 or later
 * License URI: http://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: learndash-certificates-phpword
 * Domain Path: /lang
 * Update URI: https://plugins.cleverogre.com/plugin/learndash-certificates-phpword/
 * Icon1x: https://plugins.cleverogre.com/plugin/learndash-certificates-phpword/?asset=icon-sm
 * Icon2x: https://plugins.cleverogre.com/plugin/learndash-certificates-phpword/?asset=icon
 * BannerHigh: https://plugins.cleverogre.com/plugin/learndash-certificates-phpword/?asset=banner
 * BannerLow: https://plugins.cleverogre.com/plugin/learndash-certificates-phpword/?asset=banner-sm
 * Copyright: © 2023 CleverOgre, Inc. All rights reserved.
 */

namespace LearnDash\Certificate\PhpWord;

defined('ABSPATH') || exit;

class Plugin {

    private static $data;
    public static function init() {
        self::$data = get_plugin_data(__FILE__);

        // Setup Plugin Updates
        include_once(self::get_path() . 'lib/wp-package-updater/class-wp-package-updater.php'); // Include private plugin updating library
        $package_updater = new \WP_Package_Updater(
            'https://plugins.cleverogre.com',
            wp_normalize_path(__FILE__),
            wp_normalize_path(plugin_dir_path(__FILE__)),
            false // License key not necessary
        );

        // Set Text Domain
        load_plugin_textdomain(self::get_textdomain(), false, plugin_basename(dirname(__FILE__)) . '/lang');

        // Requirements Check
        add_action('plugins_loaded', [__CLASS__, 'requirements_check'], 1, 0);

        // OgreCore Packages
        add_filter('ogre/packages', [__CLASS__, 'packages'], 10, 1);

        // Delay loading
        add_action('plugins_loaded', [__CLASS__, 'load'], 10, 0);
    }

    private static $requirements;
    static function requirements_check() {
        // Check for plugin requirements
        self::$requirements = [
            'ogrecore' => [
                'name' => __('OgreCore', self::get_textdomain()),
                'url' => 'https://plugins.cleverogre.com/plugin/ogrecore/',
                'valid' => (function_exists('is_plugin_active') && is_plugin_active('OgreCore/ogrecore.php')) || class_exists('\OgreCore\Plugin'),
            ],
            'learndash' => [
                'name' => __('LearnDash', self::get_textdomain()),
                'url' => 'https://woocommerce.com/',
                'valid' => (function_exists('is_plugin_active') && is_plugin_active('woocommerce/woocommerce.php')) || class_exists('\woocommerce'),
            ],
        ];

        $invalid = false;
        foreach (self::$requirements as $key => $data) {
            if (!!$data['valid']) continue;

            add_action('admin_notices', function () use ($data) {
                $message = sprintf(
                    __('In order to use the %1$s plugin, it is required that you install and activate the %3$s plugin. You can do this on the <a href="%2$s">plugins</a> page when logged in as an administrator. To download this plugin, visit the <a href="%4$s" target="_blank">%3$s website</a>.', self::get_textdomain()),
                    get_plugin_data(__FILE__)['Name'],
                    esc_url(admin_url('plugins.php')),
                    esc_html($data['name']),
                    esc_url($data['url'])
                );
                printf('<div class="%s"><p>%s</p></div>', esc_attr('notice notice-error'), wpautop(wp_kses_post($message)));
            });

            $invalid = true;
        }

        if (!!$invalid) remove_action('plugins_loaded', [__CLASS__, 'load'], 10, 0);
    }

    static function packages($packages) {
        if (!in_array('field', $packages)) $packages[] = 'field';
        return $packages;
    }

    static function load() {
        include_once(self::get_path() . 'inc/trait-singleton.php');
        include_once(self::get_path() . 'inc/abstract-settings.php');
        include_once(self::get_path() . 'inc/abstract-converter.php');
        include_once(self::get_path() . 'inc/class-unoconv-converter.php');
        include_once(self::get_path() . 'inc/class-dompdf-converter.php');
        include_once(self::get_path() . 'inc/class-template-values.php');
        include_once(self::get_path() . 'inc/class-template-generator.php');
        include_once(self::get_path() . 'inc/class-certificate-settings.php');
        include_once(self::get_path() . 'inc/class-display.php');
    }

    // Plugin File Path Calculations
    public static function get_path($file = __FILE__) {
        return trailingslashit(dirname($file));
    }
    public static function get_dir($file = __FILE__) {
        $dir = trailingslashit(dirname($file));
        $count = 0;

        // Sanitize for Win32 installs
        $dir = str_replace('\\', '/', $dir);

        // If file is in plugins folder
        $wp_plugin_dir = str_replace('\\', '/', WP_PLUGIN_DIR);
        $dir = str_replace($wp_plugin_dir, plugins_url(), $dir, $count);

        if ($count < 1) {
            // If file is in wp-content folder
            $wp_content_dir = str_replace('\\', '/', WP_CONTENT_DIR);
            $dir = str_replace($wp_content_dir, content_url(), $dir, $count);
        }

        if ($count < 1) {
            // If file is in ??? folder
            $wp_dir = str_replace('\\', '/', ABSPATH);
            $dir = str_replace($wp_dir, site_url('/'), $dir);
        }

        return $dir;
    }
    public static function get_hook($file = __FILE__) {
        return basename(dirname($file)) . '/' . basename($file);
    }

    // Plugin Data Accessors
    public static function get($key) {
        return isset(self::$data[$key]) ? self::$data[$key] : null;
    }
    public static function get_textdomain() {
        return self::get('TextDomain');
    }
    public static function get_id() {
        return self::get('TextDomain');
    }
    public static function get_version() {
        return self::get('Version');
    }
    public static function get_title() {
        return __(self::get('Name'), self::get_textdomain());
    }
    public static function get_short_title() {
        return __('LearnDash Certificates', self::get_textdomain());
    }
    public static function get_description() {
        return __(self::get('Description'), self::get_textdomain());
    }

}

Plugin::init();
