<?php
/**
 * @package CleverOgre
 * @subpackage FareHarbor
 * @version 0.1.0
 * @since 0.1.0
 */

namespace Ogre\FareHarbor;

if (!defined('ABSPATH')) exit;

register_deactivation_hook(__FILE__, ['\Ogre\FareHarbor\Sync', 'deactivate']);

class Sync {

    // Manual Sync

    public static function load() {
        add_action('ogre/fareharbor/settings_section', [__CLASS__, 'settings_section_button']);
        add_action('admin_post_ogre_fareharbor_sync', [__CLASS__, 'manual_synchronize']);
    }

    static function settings_section_button() {
        printf(
            '<p><a class="button-secondary" href="%s" title="%s">%s</a></p>',
            wp_nonce_url(add_query_arg([
                'action' => 'ogre_fareharbor_sync',
            ], admin_url('admin-post.php')), 'ogre_fareharbor_sync', '_fhnonce'),
            esc_attr__('Test your api settings and pre-populate your FareHarbor items', 'fareharbor-api'),
            esc_html__('Run Manual Api Synchronization', 'fareharbor-api')
        );
    }

    static function manual_synchronize() {
        if (!isset($_GET['_fhnonce']) || !wp_verify_nonce($_GET['_fhnonce'], 'ogre_fareharbor_sync')) {
            wp_die(__('The link you followed has expired. Please try again.', 'fareharbor-api'));
            exit;
        }

        if (empty(Api::get_app_key())) {
            wp_die(sprintf(
                __('Could not synchronize. App Key is not provided. Go to the <a href="%s">FareHarbor settings page</a> to insert the App Key provided by FareHarbor.', 'fareharbor-api'),
                esc_url(add_query_arg([
                    'page' => 'fareharbor/fareharbor.php',
                ], admin_url('options-general.php')))
            ));
            exit;
        }

        if (empty(Api::get_user_key())) {
            wp_die(sprintf(
                __('Could not synchronize. User Key is not provided. Go to the <a href="%s">FareHarbor settings page</a> to insert the User Key provided by FareHarbor.', 'fareharbor-api'),
                esc_url(add_query_arg([
                    'page' => 'fareharbor/fareharbor.php',
                ], admin_url('options-general.php')))
            ));
            exit;
        }

        if (!wp_http_validate_url(Api::get_root_url())) {
            wp_die(sprintf(
                __('Could not synchronize. Root URL is invalid. Go to the <a href="%s">FareHarbor settings page</a> to correct this issue.', 'fareharbor-api'),
                esc_url(add_query_arg([
                    'page' => 'fareharbor/fareharbor.php',
                ], admin_url('options-general.php')))
            ));
            exit;
        }

        $result = self::synchronize();
        if (!$result) {
            wp_die(__('FareHarbor API synchronization failed for an unknown reason.', 'fareharbor-api'));
            exit;
        }

        wp_die(sprintf(
            __('Congrats! You\'ve successfully synchronized all items with your FareHarbor account. <a href="%s">Return to the FareHarbor settings page.</a>', 'fareharbor-api'),
            esc_url(add_query_arg([
                'page' => 'fareharbor/fareharbor.php',
            ], admin_url('options-general.php')))
        ));
        exit;
    }

    // Settings

    private static function is_enabled() {
        return apply_filters('ogre/fareharbor/api_sync_enabled', Settings::get_option('fh_api_sync_enabled', false) === 'on');
    }
    private static function get_shortname() {
        return apply_filters('ogre/fareharbor/api_sync_shortname', Settings::get_option('fh_api_sync_shortname', ''));
    }

    // Cron Scheduling

    public static function schedule() {
        add_action('fareharbor_sync', [__CLASS__, 'synchronize']);
        if (!wp_next_scheduled('fareharbor_sync') && self::is_enabled()) {
            wp_schedule_event(time(), 'daily', 'fareharbor_sync');
        } else if (!!wp_next_scheduled('fareharbor_sync') && !self::is_enabled()) {
            wp_clear_scheduled_hook('fareharbor_sync');
        }
    }

    public static function deactivate() {
        $timestamp = wp_next_scheduled('fareharbor_sync');
        wp_unschedule_event($timestamp, 'fareharbor_sync');
    }

    // API Access

    private static function get_shortnames() {
        $shortnames = [];
        $shortname = self::get_shortname();
        if (empty($companies)) {
            $companies = Api::get_companies();
            if (!$companies) return false;
            foreach ($companies as $company) {
                if (!isset($company['shortname']) || empty($company['shortname'])) continue;
                $shortnames[] = $company['shortname'];
            }
        } else {
            $shortnames = [$shortname];
        }
        return apply_filters('ogre/fareharbor/get_shortnames', !empty($shortnames) ? $shortnames : false);
    }

    private static function get_items($shortnames = false) {
        if ($shortnames === false) $shortnames = self::get_shortnames();
        if (is_string($shortnames)) $shortnames = [$shortnames];
        if (empty($shortnames)) return false;

        $items = [];
        foreach ($shortnames as $shortname) {
            $_items = Api::get_items($shortname);
            if (!is_array($_items) || empty($_items)) continue;
            $items = array_merge($items, $_items);
        }
        return apply_filters('ogre/fareharbor/get_items', !empty($items) ? $items : false, $shortnames);
    }

    // Synchronization Routines

    static function synchronize() {
        $items = self::get_items();
        if (!is_array($items) || empty($items)) return false;

        do_action('ogre/fareharbor/synchronize', $items);
        return self::update_items($items) > 0;
    }

    private static function update_items($items) {
        do_action('ogre/fareharbor/update_items', $items);
        return count(array_filter(array_map([__CLASS__, 'update_item'], $items)));
    }
    private static function update_item($item) {
        do_action('ogre/fareharbor/update_item', $item);
        return Item::update_by_data($item);
    }

}

Sync::load();
Sync::schedule();
