<?php
/**
 * @package WooCommerce
 * @subpackage NewtekGateway
 * @version 1.2.0
 * @since 1.2.0
 */

namespace WooCommerce\NewtekGateway;

defined('ABSPATH') || exit;

class PaymentGateway extends \WC_Payment_Gateway_CC {

    function __construct() {
        // Standard Variables for Gateway
        $this->id = Plugin::get_id();
        $this->icon = trailingslashit(Plugin::get_dir()) . 'assets/img/credit-cards.png';
        $this->has_fields = true;

        if (!is_array($this->supports)) $this->supports = [];
        $this->supports = array_merge($this->supports, [
            'default_credit_card_form',
            'products',
            /*
            'subscriptions',
            'subscription_cancellation',
            'subscription_suspension',
            'subscription_reactivation',
            'subscription_amount_changes',
            'subscription_date_changes',
            'multiple_subscriptions',
            */
        ]);

        $this->method_title = Plugin::get_short_title();
        $this->method_description = Plugin::get_description();

        // Initialize Payment Gateway settings
        $this->init_form_fields();
        $this->init_settings();

        // User set variables
        $this->title = $this->get_option('title');
        $this->description = $this->get_option('description');

        // Newtek Gateway Settings
        $this->gateway_settings = [
            'api_key' => $this->get_option('api_key'),
            'api_pin' => $this->get_option('api_pin'),
            'use_sandbox' => $this->get_option('use_sandbox') == 'yes' ? true : false,
            'description' => $this->get_option('payment_description'),
            'hold' => $this->get_option('payment_hold') == 'yes' ? true : false,
        ];

        // Setup hook for updating settings
        add_action('woocommerce_update_options_payment_gateways_' . $this->id, [$this, 'process_admin_options']);
        //add_action('woocommerce_scheduled_subscription_payment_' . $this->id, [$this, 'process_subscription_payment']);
    }

    function init_form_fields() {
        $this->form_fields = [
            // Default Settings
            'enabled' => [
                'title' => __('Enable/Disable', Plugin::get_textdomain()),
                'type' => 'checkbox',
                'label' => 'Enable ' . $this->method_title,
                'default' => 'yes'
            ],
            'title' => [
                'title' => __('Title', Plugin::get_textdomain()),
                'type' => 'text',
                'description' => __('This controls the title which the user sees during checkout.', Plugin::get_textdomain()),
                'default' => $this->method_title,
                'desc_tip' => true
            ],
            'description' => [
                'title' => __('Customer Message', Plugin::get_textdomain()),
                'type' => 'textarea',
                'default' => ''
            ],

            // Gateway Settings
            'api_key' => [
                'title' => __('API Key', Plugin::get_textdomain()),
                'type' => 'text',
                'description' => '',
                'default' => '',
                'desc_tip' => false
            ],
            'api_pin' => [
                'title' => __('API Pin', Plugin::get_textdomain()),
                'type' => 'text',
                'description' => '',
                'default' => '',
                'desc_tip' => false
            ],
            'use_sandbox' => [
                'title' => __('Enable Sandbox Mode?', Plugin::get_textdomain()),
                'type' => 'checkbox',
                'label' => __('Enables sandbox for testing purposes', Plugin::get_textdomain()),
                'default' => 'no'
            ],
            'payment_description' => [
                'title' => __('Payment Description', Plugin::get_textdomain()),
                'type' => 'text',
                'description' => '',
                'default' => '',
                'desc_tip' => false
            ],
            'payment_hold' => [
                'title' => __('Mark Payment as On Hold when Complete?', Plugin::get_textdomain()),
                'type' => 'checkbox',
                'label' => __('Instead of setting payment as complete in WooCommerce upon processing, marks payment as On Hold until admin marks it as completed.', Plugin::get_textdomain()),
                'default' => 'no'
            ],
        ];
    }

    function payment_fields() {
        $this->form();
    }

    function validate_fields() {
        return true;
    }

    private function get_request_field(array|string $keys):string|bool {
        if (is_string($keys)) $keys = [$keys];
        $value = false;
        foreach ($keys as $key) {
            if (!is_string($key) || empty($key) || !isset($_POST[$key]) || !is_string($_POST[$key]) || empty($_POST[$key])) continue;
            $value = $_POST[$key];
            break;
        }
        return $value;
    }
    private function get_request_card_number():string|bool {
        $value = $this->get_request_field([$this->id . '-card-number', 'ccnumber']);
        if (!is_string($value) || empty($value)) return false;
        return str_replace(' ', '', $value);
    }
    private function get_request_card_cvc():string|bool {
        return $this->get_request_field([$this->id . '-card-cvc', 'cccvc']);
    }
    private function get_request_card_expiry():string|bool {
        $value = $this->get_request_field([$this->id . '-card-expiry', 'ccexpiry']);
        if (!is_string($value) || empty($value)) return false;
        return str_replace(['/', ' '], '', $value);
    }
    private function get_request_card_data():array|bool {
        $data = [
            'card' => $this->get_request_card_number(),
            'cvc' => $this->get_request_card_cvc(),
            'exp_date' => $this->get_request_card_expiry(),
        ];
        if (count(array_filter($data)) < count($data)) return false;
        return $data;
    }

    function build_transaction_data($order, $amount = false) {
        if ($amount === false) $amount = $order->get_total();
        if (!is_numeric($amount)) return false;

        $geo = new \WC_GeoLocation();

        $card = $this->get_request_card_data();
        if (!$card) return false;

        $transaction = [
            // API Settings
            'key' => $this->gateway_settings['api_key'],
            'pin' => $this->gateway_settings['api_pin'],
            'usesandbox' => $this->gateway_settings['use_sandbox'],
            'ip' => (string)$order->get_user_id(),
            'testmode' => 0,
            'command' => 'cc:sale',

            // Transaction Settings
            'card' => $card['card'],
            'exp' => $card['exp_date'],
            'amount' => number_format(floatval($amount), 2, '.', ''),
            'invoice' => $order->get_transaction_id(),
            'orderid' => $order->get_id(),
            'cardholder' => trim($order->get_billing_first_name() . ' ' . $order->get_billing_last_name()),
            'street' => trim($order->get_billing_address_1() . ' ' . $order->get_billing_address_2()),
            'zip' => $order->get_billing_postcode(),
            'description' => $this->gateway_settings['description'],
            'cvv2' => $card['cvc'],
        ];

        return $transaction;
    }

    private function process_transaction($order_id, $amount = false) {
        $order = is_a($order_id, '\WC_Order') ? $order_id : new \WC_Order($order_id);

        if (!class_exists('\umTransaction')) include_once(trailingslashit(Plugin::get_path()) . 'vendor/usaepay.php');

        try {
            $data = $this->build_transaction_data($order, $amount);
            if ($data === false || !is_array($data) || empty($data)) {
                ob_start();
                var_dump($_POST);
                $output = ob_get_clean();
                throw new \Exception(__('Could not retrieve card data.', Plugin::get_textdomain()));
            }

            $transaction = new \umTransaction();

            // Transaction API Settings
            $transaction->key = $data['key'];
            $transaction->pin = $data['pin'];
            $transaction->usesandbox = isset($data['sandbox']) ? $data['sandbox'] : false;
            $transaction->ip = $data['ip'];
            $transaction->testmode = $data['testmode'];

            // Type of Transaction
            $transaction->command = $data['command'];

            // Order Information
            $transaction->card = $data['card'];
            $transaction->exp = $data['exp'];
            $transaction->amount = $data['amount'];
            $transaction->invoice = $data['invoice'];
            $transaction->orderid = $data['orderid'];
            $transaction->cardholder = $data['cardholder'];
            $transaction->street = $data['street'];
            $transaction->zip = $data['zip'];
            $transaction->description = $data['description'];
            $transaction->cvv2 = $data['cvv2'];

            if (!$transaction->Process()) {
                throw new \Exception(sprintf(__('Card declined (%s). <br />Reason: %s', Plugin::get_textdomain()), $transaction->result, $transaction->error));
            }
        } catch (\Exception $e) {
            return $e;
        }

        return true;
    }

    function process_payment($order_id) {
        global $woocommerce;
        $order = is_a($order_id, '\WC_Order') ? $order_id : new \WC_Order($order_id);

        $result = $this->process_transaction($order);

        if ($result !== true) {
            if (is_a($result, '\Exception')) {
                wc_add_notice(sprintf(__('Payment transaction error: %s', Plugin::get_textdomain()), $result->getMessage()), 'error');
            } else {
                wc_add_notice(__('Unknown payment transaction error.', Plugin::get_textdomain()), 'error');
            }
            return [
                'result' => 'error',
                'redirect' => wc_get_checkout_url(),
            ];
        }

        // Add successful payment notice
        wc_add_notice(__('Payment successful.', Plugin::get_textdomain()), 'success');

        if ($this->gateway_settings['hold'] == true) {
            // Mark payment as On-Hold
            $order->update_status('on-hold', sprintf(__('%s payment processed.', Plugin::get_textdomain()), $this->method_title));
        } else {
            // Mark payment as complete when finished processing
            $order->payment_complete();
        }

        // Reduce stock levels
        $order->reduce_order_stock();

        // Remove cart
        $woocommerce->cart->empty_cart();

        return [
            'result' => 'success',
            'redirect' => $this->get_return_url($order),
        ];
    }

    public function process_subscription_payment($amount, $order_id) {
        $order = is_a($order_id, '\WC_Order') ? $order_id : new \WC_Order($order_id);

        $result = $this->process_transaction($order, $amount);

        if ($result !== true) {
            if (is_a($result, '\Exception')) {
                wc_add_notice(sprintf(__('Subscription payment transaction error: %s', Plugin::get_textdomain()), $e->getMessage()), 'error');
            } else {
                wc_add_notice(__('Unknown subscription payment transaction error.', Plugin::get_textdomain()), 'error');
            }
            // NOTE: Not sure if failing the order/subscription is necessary.
            $order->payment_failed();
            return false;
        }

        // Add successful payment notice
        wc_add_notice(sprintf(__('Subscription payment of %s successful.', Plugin::get_textdomain()), is_numeric($amount) ? number_format(floatval($amount), 2, '.', '') : __('(unknown amount)', Plugin::get_textdomain())), 'success');

        if ($this->gateway_settings['hold'] == true && $order->get_status() != 'on-hold') {
            // Mark payment as On-Hold
            $order->update_status('on-hold', sprintf(__('%s payment processed.', Plugin::get_textdomain()), $this->method_title));
        } else {
            // Mark payment as complete when finished processing
            $order->payment_complete();
        }

        // Reduce stock levels
        // NOTE: Not sure if stock levels should be reduced with subscription payment
        $order->reduce_order_stock();

        // NOTE: Not sure of how to return from this action.
        return true;
    }

}
