<?php
/**
 * Plugin Name: WooCommerce Newtek Payment Gateway Add-On
 * Plugin URI: https://plugins.cleverogre.com/plugin/woocommerce-gateway-newtek/
 * Description: Extends WooCommerce with the Newtek Gateway payment gateway.
 * Version: 1.2.0
 * Requires at Least: 6.0
 * Requires PHP: 8.0
 * Author: CleverOgre
 * Author URI: https://cleverogre.com/
 * License: GPLv2 or later
 * License URI: http://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: woocommerce-gateway-newtek
 * Domain Path: /lang
 * Update URI: https://plugins.cleverogre.com/plugin/woocommerce-gateway-newtek/
 * Icon1x: https://plugins.cleverogre.com/plugin/woocommerce-gateway-newtek/?asset=icon-sm
 * Icon2x: https://plugins.cleverogre.com/plugin/woocommerce-gateway-newtek/?asset=icon
 * BannerHigh: https://plugins.cleverogre.com/plugin/woocommerce-gateway-newtek/?asset=banner
 * BannerLow: https://plugins.cleverogre.com/plugin/woocommerce-gateway-newtek/?asset=banner-sm
 * Copyright: © 2023 CleverOgre, Inc. All rights reserved.
 */

namespace WooCommerce\NewtekGateway;

defined('ABSPATH') || exit;

class Plugin {

    private static $data;
    public static function init() {
        self::$data = get_plugin_data(__FILE__);

        // Setup Plugin Updates
        include_once(self::get_path() . 'vendor/wp-package-updater/class-wp-package-updater.php'); // Include private plugin updating library
        $package_updater = new \WP_Package_Updater(
            'https://plugins.cleverogre.com',
            wp_normalize_path(__FILE__),
            wp_normalize_path(plugin_dir_path(__FILE__)),
            false // License key not necessary
        );

        // Set Text Domain
        load_plugin_textdomain(self::get_textdomain(), false, plugin_basename(dirname(__FILE__)) . '/lang');

        // Requirements Check
        add_action('plugins_loaded', [__CLASS__, 'requirements_check'], 1, 0);

        // Delay loading
        add_action('plugins_loaded', [__CLASS__, 'load'], 11, 0);

        // Register with standard WC payment gateways
        add_filter('woocommerce_payment_gateways', [__CLASS__, 'register_gateway']);

        // Register with WooCommerce Blocks integration
		add_action('woocommerce_blocks_loaded', [__CLASS__, 'register_block_support']);
    }

    private static $requirements;
    static function requirements_check() {
        // Check for plugin requirements
        self::$requirements = [
            'woocommerce' => [
                'name' => __('WooCommerce', self::get_textdomain()),
                'url' => 'https://woocommerce.com/',
                'valid' => (function_exists('is_plugin_active') && is_plugin_active('woocommerce/woocommerce.php')) || class_exists('\woocommerce'),
            ],
        ];

        $invalid = false;
        foreach (self::$requirements as $key => $data) {
            if (!!$data['valid']) continue;

            add_action('admin_notices', function () use ($data) {
                $message = sprintf(
                    __('In order to use the %1$s plugin, it is required that you install and activate the %3$s plugin. You can do this on the <a href="%2$s">plugins</a> page when logged in as an administrator. To download this plugin, visit the <a href="%4$s" target="_blank">%3$s website</a>.', self::get_textdomain()),
                    get_plugin_data(__FILE__)['Name'],
                    esc_url(admin_url('plugins.php')),
                    esc_html($data['name']),
                    esc_url($data['url'])
                );
                printf('<div class="%s"><p>%s</p></div>', esc_attr('notice notice-error'), wpautop(wp_kses_post($message)));
            });

            $invalid = true;
        }

        if (!!$invalid) remove_action('plugins_loaded', [__CLASS__, 'load'], 11, 0);
    }

    // Load assets
    public static function load() {
        include_once(self::get_path() . 'includes/trait-singleton.php');

        if (class_exists('\WC_Payment_Gateway')) {
            include_once(self::get_path() . 'includes/class-payment-gateway.php');
            include_once(self::get_path() . 'includes/class-rest-process-payment.php');
        }
    }

    // Register with standard WC payment gateways
    public static function register_gateway($methods) {
        $methods[] = '\WooCommerce\NewtekGateway\PaymentGateway';
        return $methods;
    }

    // Register with WooCommerce Blocks integration
    public static function register_block_support() {
        if (!class_exists('\Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType')) return;
        include_once(self::get_path() . 'includes/class-block-support.php');
		add_action('woocommerce_blocks_payment_method_type_registration', [__CLASS__, 'register_block_payment_method']);
    }
    public static function register_block_payment_method(\Automattic\WooCommerce\Blocks\Payments\PaymentMethodRegistry $payment_method_registry) {
        if (!class_exists('\WooCommerce\NewtekGateway\BlockSupport')) return;
        $payment_method_registry->register(new BlockSupport());
    }

    // Plugin File Path Calculations
    public static function get_file():string {
        return __FILE__;
    }
    public static function get_path($file = __FILE__):string {
        return trailingslashit(dirname($file));
    }
    public static function get_dir($file = __FILE__):string {
        $dir = trailingslashit(dirname($file));
        $count = 0;

        // Sanitize for Win32 installs
        $dir = str_replace('\\', '/', $dir);

        // If file is in plugins folder
        $wp_plugin_dir = str_replace('\\', '/', WP_PLUGIN_DIR);
        $dir = str_replace($wp_plugin_dir, plugins_url(), $dir, $count);

        if ($count < 1) {
            // If file is in wp-content folder
            $wp_content_dir = str_replace('\\', '/', WP_CONTENT_DIR);
            $dir = str_replace($wp_content_dir, content_url(), $dir, $count);
        }

        if ($count < 1) {
            // If file is in ??? folder
            $wp_dir = str_replace('\\', '/', ABSPATH);
            $dir = str_replace($wp_dir, site_url('/'), $dir);
        }

        return $dir;
    }
    public static function get_hook($file = __FILE__):string {
        return basename(dirname($file)) . '/' . basename($file);
    }

    // Plugin Data Accessors
    public static function get($key) {
        return isset(self::$data[$key]) ? self::$data[$key] : null;
    }
    public static function get_textdomain() {
        return self::get('TextDomain');
    }
    public static function get_id($append = '') {
        $id = self::get('TextDomain');
        if (!empty($append)) $id .= '_' . sanitize_title($append);
        return self::get('TextDomain');
    }
    public static function get_version() {
        return self::get('Version');
    }
    public static function get_title() {
        return __(self::get('Name'), self::get_textdomain());
    }
    public static function get_short_title() {
        return __('Newtek Gateway', self::get_textdomain());
    }
    public static function get_description() {
        return __(self::get('Description'), self::get_textdomain());
    }

    // Template Parts
    private static function locate_template(string|array $templates, bool $load = false, bool $require_once = true, array $args = []):string {
        $located = '';
        $dirs = [
            rtrim(STYLESHEETPATH, '/') . '/woocommerce-bistrack/',
            rtrim(TEMPLATEPATH . '/') . '/woocommerce-bistrack/',
            rtrim(self::get_path(), '/') . '/template-parts/',
        ];
        foreach ((array)$templates as $template) {
            if (!is_string($template) || empty($template)) continue;
            foreach ($dirs as $dir) {
                if (!file_exists(rtrim($dir, '/') . '/' . $template)) continue;
                $located = rtrim($dir, '/') . '/' . $template;
                break;
            }
            if (!empty($located)) break;
        }

        if (!!$load && !empty($located)) load_template($located, $require_once, $args);
        return $located;
    }
    private static function get_templates(string $template_name, string $part_name = null):array {
        $templates = [];
        if (!is_null($part_name) && !empty($part_name)) {
            $templates[] = "{$template_name}-{$part_name}.php";
        }
        $templates[] = "{$template_name}.php";
        return $templates;
    }
    public static function get_template_part(string $template_name, string $part_name = null, array $args = []):string|bool {
        $located = self::locate_template(self::get_templates($template_name, $part_name), true, false, $args);
        return !!$located ? $located : false;
    }
    public static function load_template_part(string $template_name, string $part_name = null, array $args = []):string {
        ob_start();
        self::get_template_part($template_name, $part_name, $args);
        return ob_get_clean();
    }
    public static function include_template_part(array $data, string $template_name, string $part_name = null, array $args = []):string {
        ob_start();
        extract($data);
        include(self::locate_template(self::get_templates($template_name, $part_name), false, false, $args));
        return ob_get_clean();
    }

}

Plugin::init();
