/**
 * @package WooCommerce
 * @subpackage NewtekGateway
 * @version 1.2.0
 * @since 1.2.0
 */

// External
import { useRef, Component } from 'react';
import { __ } from '@wordpress/i18n';
import { registerPaymentMethod } from '@woocommerce/blocks-registry';
import { decodeEntities } from '@wordpress/html-entities';
import { getSetting } from '@woocommerce/settings';
import { useState, useEffect, Fragment } from '@wordpress/element';

// Internal
import { PAYMENT_METHOD_NAME, PAYMENT_METHOD_TEXTDOMAIN } from './constants.js';
import Fields from './fields.js';

// Settings
const settings = getSetting(PAYMENT_METHOD_NAME + '_data', {});

const defaultLabel = __(
	'Newtek Gateway',
	PAYMENT_METHOD_TEXTDOMAIN
);
const label = decodeEntities(settings.title) || defaultLabel;

/**
 * Label component
 *
 * @param {*} props Props from payment API.
 */
const Label = (props) => {
	const { PaymentMethodLabel } = props.components;
	const src = settings.assets_url + 'img/credit-cards.png';
	return (
		<>
			<PaymentMethodLabel text={ label } />
			<img
				src={ src }
				alt={ label }
				style={{marginLeft: '0.5em'}}
			/>
		</>
	);
};

/**
 * Description component
 */
const Description = (props) => {
	return decodeEntities(settings.description || '');
};

/**
 * Content component
 */
const Content = (props) => {

	const { eventRegistration, emitResponse } = props;
	const { onPaymentProcessing } = eventRegistration;

	const fields = useRef(null);

	useEffect( () => {
		const unsubscribe = onPaymentProcessing( async () => {
			let valid = true;

			if (!fields.current || !fields.current.state.number || fields.current.state.number.length < 13) valid = false;
			if (!fields.current || !fields.current.state.expiry || fields.current.state.expiry.length < 4) valid = false;
			if (!fields.current || !fields.current.state.cvc || fields.current.state.cvc.length < 3) valid = false;

			if (!!valid) {
				return {
					type: emitResponse.responseTypes.SUCCESS,
					meta: {
						paymentMethodData: {
							ccNumber: fields.current.state.number,
							ccExpiry: fields.current.state.expiry,
							ccCvc: fields.current.state.cvc,
						},
					}
				};
			}

			return {
				type: emitResponse.responseTypes.ERROR,
				message: __( 'Your credit card details are invalid. Please adjust your credit card information and try again.', PAYMENT_METHOD_TEXTDOMAIN ),
			}
		} );
		return () => {
			unsubscribe();
		};
	}, [
		emitResponse.responseTypes.ERROR,
		emitResponse.responseTypes.SUCCESS,
		onPaymentProcessing,
		fields,
	] );

	return (
		<Fragment>
			<Description />
			<Fields ref={ fields } />
		</Fragment>
	);
};
/**
 * Payment method config object.
 */
const WCNewtekGateway = {
	name: PAYMENT_METHOD_NAME,
	label: <Label />,
	content: <Content />,
	edit: <Content />,
	canMakePayment: () => true,
	ariaLabel: label,
	supports: {
		features: settings.supports,
	},
};

registerPaymentMethod(WCNewtekGateway);
