<?php
/**
 * @package CleverOgre
 * @subpackage OgreModal
 * @version 0.4.2
 * @since 0.1.0
 */

namespace OgreModal;

if (!defined('ABSPATH')) exit;

if (!defined('SECONDS_IN_DAY')) {
    define('SECONDS_IN_DAY', 24 * 60 * 60);
}

use WP_Block_Type_Registry;
use WP_HTML_Tag_Processor;

class Api {

    private static function is_user_logged_in() {
    	$user = wp_get_current_user();
    	if (empty($user->ID)) return false;
    	return true;
    }

    public static function validate_modal($post_id) {
        $valid = true;

        if (is_admin()) $valid = false;
        if (get_post_type($post_id) != 'modal') $valid = false;

        if (!in_array(get_post_status($post_id), ['publish', 'private'])) $valid = false;
        if (get_post_status($post_id) == 'private' && (!is_user_logged_in() || !current_user_can('read_private_posts'))) $valid = false;

        if (!empty(get_field('limit', $post_id))) {
            $ids = array_map('intval', get_field('limit', $post_id));
            global $post;
            if (is_front_page() && !in_array(get_option('page_on_front'), $ids)) $valid = false;
            if (is_home() && !in_array(get_option('page_for_poss'), $ids)) $valid = false;
            if (is_singular() && (is_null($post) || !is_a($post, 'WP_Post') || !in_array($post->ID, $ids))) $valid = false;
        }

        return apply_filters('ogremodal/validate_modal', $valid, $post_id);
    }

    public static function get_modal_ids() {
        return apply_filters('ogremodal/get_modal_ids', get_posts(apply_filters('ogremodal/modal_query_args', [
            'post_type' => 'modal',
            'posts_per_page' => -1,
            'offset' => 0,
            'fields' => 'ids',
            'post_status' => ['publish', 'private'],
        ])));
    }

    public static function get_valid_modal_ids() {
        $modal_ids = array_filter(self::get_modal_ids(), [__CLASS__, 'validate_modal']);
        return apply_filters('ogremodal/get_valid_modal_ids', !empty($modal_ids) ? $modal_ids : false);
    }

    public static function has_valid_modals() {
        return apply_filters('ogremodal/has_valid_modals', !empty(self::get_valid_modal_ids()));
    }

    private static function load_template($filename, $filepart = '', $echo = true, $vars = array()) {
        $path = Settings::get_option('path') . 'templates/' . $filename . '-' . $filepart . '.php';
        if (!file_exists($path)) {
            $path = Settings::get_option('path') . 'templates/' . $filename . '.php';
            if (!file_exists($path)) {
                if ($echo) {
                    echo '';
                    return false;
                } else {
                    return '';
                }
            }
        }

        ob_start();
        extract($vars);
        include($path);
        $html = ob_get_contents();
        ob_end_clean();

        if ($echo) {
            echo $html;
            return true;
        } else {
            return $html;
        }
    }

    public static function load_modal($post_id, $echo = true) {
        return self::load_template('modal', '', $echo, [
            'post_id' => $post_id,
        ]);
    }

    public static function load_modals($echo = true) {
        $ids = self::get_valid_modal_ids();
        if (!is_array($ids) || empty($ids)) return false;
        return array_map(function ($post_id) use ($echo) {
            return self::load_modal($post_id, $echo);
        }, $ids);
    }

    public static function parse_hex_color($hex) {
        // Check if empty
        if (empty($hex)) return false;

        // Validate with Regex
        $regex = '/^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/';
        if (preg_match($regex, $hex) != 1) return false;

        $dec = array('r' => 0, 'g' => 0, 'b' => 0);

        if (strlen($hex) == strlen('#000')) {
            $dec['r'] = hexdec(substr($hex, 1, 1) . substr($hex, 1, 1));
            $dec['g'] = hexdec(substr($hex, 2, 1) . substr($hex, 1, 1));
            $dec['b'] = hexdec(substr($hex, 3, 1) . substr($hex, 1, 1));
        } else {
            $dec['r'] = hexdec(substr($hex, 1, 2));
            $dec['g'] = hexdec(substr($hex, 3, 2));
            $dec['b'] = hexdec(substr($hex, 5, 2));
        }

        return $dec;
    }

    public static function parse_dismiss_duration($duration) {
        if (is_numeric($duration)) return intval($duration);

        if (!is_string($duration)) $duration = 'hour';

        switch ($duration) {
            case 'immediate':
                return 0;
            case 'minute':
                return MINUTE_IN_SECONDS;
            case 'hour':
                return HOUR_IN_SECONDS;
            case 'day':
                return DAY_IN_SECONDS;
            case 'week':
                return WEEK_IN_SECONDS;
            case 'month':
                return MONTH_IN_SECONDS;
            case 'year':
                return YEAR_IN_SECONDS;
            default:
                return $duration;
        }
    }

    public static function get_field_group($title) {
        global $wpdb;

        // Check public field groups
        $field_group = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$wpdb->posts} WHERE post_type='%s' AND post_status='%s' AND post_title='%s';", 'acf-field-group', 'publish', $title));
        if (!is_null($field_group) && is_object($field_group)) {
            return acf_get_field_group($field_group->post_name);
        }

        // Check private field groups
        $field_group = acf_get_local_store('groups')->query([
            'title' => $title,
        ]);
        $field_group = array_shift($field_group);
        if (!is_null($field_group) && !empty($field_group)) return $field_group;

        return false;
    }

    public static function field_group_exists($value, $type = 'post_title') {
        return !!self::get_field_group($value, $type);
    }

    public static function get_settings($post_id = false) {
        // Get global ACF field group settings
        $data = self::get_field_group_data('OgreModal Global Settings', false, 'ogremodal_');

        // Get individual post settings
        if (is_a($post_id, 'WP_Post')) $post_id = $post_id->ID;
        if (is_numeric($post_id) && get_post_type(intval($post_id)) == 'modal') {
            $data = array_merge($data, self::get_field_group_data('Modal Configuration', $post_id));
            // Ensure that id is set
            $data['post_id'] = $post_id;
            if (!isset($data['id']) || empty($data['id'])) $data['id'] = self::get_id($post_id);
        }

        return apply_filters('ogremodal/get_settings', $data);
    }

    private static function get_field_group_data($name, $post_id = false, $prefix = '') {
        $field_group = self::get_field_group($name);
        if (empty($field_group)) return false;

        $fields = acf_get_fields($field_group);
        $data = array();
        foreach ($fields as $field) {
            if (in_array($field['type'], array('tab'))) continue;
            if (!empty($prefix) && !strpos($field['name'], $prefix) === 0) continue;

            $field['name'] = substr($field['name'], strlen($prefix));
            if ($post_id == false) $field['value'] = get_field($field['key'], 'options');
            else $field['value'] = get_field($field['key'], $post_id);

            switch ($field['type']) {
                case 'number':
                    $field['value'] = is_numeric($field['value']) ? floatval($field['value']) : $field['value'];
                    break;
                case 'range':
                    $field['value'] = is_numeric($field['value']) ? intval($field['value']) : $field['value'];
                    break;
            }

            $data[$field['name']] = $field['value'];
        }

        return $data;
    }

    private static function generate_id() {
        return wp_generate_uuid4();
    }

    public static function get_id($post_id) {
        $id = get_field('id', $post_id);
        if (empty($id)) {
            $id = self::generate_id();
            update_field('id', $id, $post_id);
        }
        return $id;
    }

    // General Functions (from OgreCore)

    // Run an action if it's already been called, otherwise schedule it. Doesn't support arguments.
    public static function maybe_add_action(string $tag, callable $function_to_add, int $priority = 10) {
        if (did_action($tag) <= 0) {
            add_action($tag, $function_to_add, $priority);
        } else if (is_callable($function_to_add)) {
            call_user_func($function_to_add);
        }
    }

    // Replacement for deprecated get_page_by_title function in WP 6.2
    private const PAGE_TITLES_TRANSIENT_KEY = 'ogre_page_titles';
    public static function get_page_by_title($page_title, $output = OBJECT, $post_type = 'page') {
        // Check cache
        $cache_key = sprintf('ogre/get_page_by_title_%s_%s', $post_type, $page_title);
        $post_id = wp_cache_get($cache_key);
        if (is_null($post_id) || $post_id === false) {

            // Check transient
            $page_titles = get_transient(self::PAGE_TITLES_TRANSIENT_KEY);
            if (!is_array($page_titles)) $page_titles = [];

            // Actually grab post_id by post_title via query
            if (!array_key_exists($page_title, $page_titles)) {
                $page_titles[$page_title] = self::_get_page_by_title($page_title, $output, $post_type);
                // Update transient
                set_transient(self::PAGE_TITLES_TRANSIENT_KEY, $page_titles, HOUR_IN_SECONDS);
            }
            $post_id = $page_titles[$page_title];

            // Update cache
            wp_cache_set($cache_key, $post_id);
        }
        return $post_id > 0 ? get_post($post_id, $output) : null;
    }
    private static function _get_page_by_title($page_title, $output = OBJECT, $post_type = 'page') {
        $post_ids = get_posts([
            'post_type'              => $post_type,
            'title'                  => $page_title,
            'post_status'            => 'all',
            'posts_per_page'         => 1,
            'no_found_rows'          => true,
            'ignore_sticky_posts'    => true,
            'update_post_term_cache' => false,
            'update_post_meta_cache' => false,
            'orderby'                => 'post_date ID',
            'order'                  => 'ASC',
            'fields'                 => 'ids',
        ]);
        return !empty($post_ids) ? absint(array_values($post_ids)[0]) : 0;
    }

    // Generate Labels Array for Post Types and Taxonomies
    public static function get_labels($plural, $singular, $hierarchical = false, $labels = array()) {
        $ogre_labels = array(
            'name' => $plural,
            'singular_name' => $singular,
            'menu_name' => $plural,
            'name_admin_bar' => $singular,
            'add_new' => __('Add New', 'ogrecore'),
            'add_new_item' => sprintf(__('Add New %s', 'ogrecore'), $singular),
            'new_item' => sprintf(__('New %s', 'ogrecore'), $singular),
            'edit_item' => sprintf(__('Edit %s', 'ogrecore'), $singular),
            'view_item' => sprintf(__('View %s', 'ogrecore'), $singular),
            'all_items' => sprintf(__('All %s', 'ogrecore'), $plural),
            'search_items' => sprintf(__('Search %s', 'ogrecore'), $plural),
            'parent_item_colon' => sprintf(__('Parent %s:', 'ogrecore'), $plural),
            'not_found' => sprintf(__('No %s found.', 'ogrecore'), strtolower($plural)),
            'not_found_in_trash' => sprintf(__('No %s found in Trash.', 'ogrecore'), strtolower($plural)),

            'update_item' => sprintf(__('Update %s', 'ogrecore'), $singular),
            'new_item_name' => sprintf(__('New %s Name', 'ogrecore'), $singular),
            'parent_item' => sprintf(__('Parent %s', 'ogrecore'), $singular),
            'popular_items' => $hierarchical ? null : sprintf(__('Popular %s', 'ogrecore'), $plural),
            'separate_items_with_commas' => $hierarchical ? null : sprintf(__('Separate %s with commas', 'ogrecore'), strtolower($plural)),
            'add_or_remove_items' => $hierarchical ? null : sprintf(__('Add or remove %s', 'ogrecore'), strtolower($plural)),
            'choose_from_most_used' => $hierarchical ? null : sprintf(__('Choose from the most used %s', 'ogrecore'), strtolower($plural)),

            'item_published' => sprintf(__('%s published.', 'ogrecore'), $singular),
            'item_published_privately' => sprintf(__('%s published privately.', 'ogrecore'), $singular),
            'item_reverted_to_draft' => sprintf(__('%s reverted to draft.', 'ogrecore'), $singular),
            'item_scheduled' => sprintf(__('%s scheduled.', 'ogrecore'), $singular),
            'item_updated' => sprintf(__('%s updated.', 'ogrecore'), $singular),
        );
        $ogre_labels = wp_parse_args($labels, $ogre_labels);

        return apply_filters('ogre/labels', $ogre_labels, $plural, $singular, $hierarchical, $labels);
    }

    private static $block_styles = '';
    private static $block_post_id = 0;
    public static function do_blocks($post_id = 0, $post_content = null) {
        self::$block_styles = '';
        if (!$post_id) $post_id = get_the_ID();
        self::$block_post_id = $post_id;
        if (is_null($post_content)) $post_content = get_the_content(null, false, $post_id);
        remove_filter( 'render_block', 'wp_render_layout_support_flag', 10, 2 );
        add_filter( 'render_block', [__CLASS__, 'render_layout_support_flag'], 10, 2 );
        $output = do_blocks($post_content);
        remove_filter( 'render_block', [__CLASS__, 'render_layout_support_flag'], 10, 2 );
        add_filter( 'render_block', 'wp_render_layout_support_flag', 10, 2 );
        if (!empty(self::$block_styles)) {
            $output .= sprintf(
                '<style type="text/css" id="ogremodal-%d-inline-css">%s</style>',
                $post_id,
                self::$block_styles
            );
        }
        return $output;
    }

    public static function render_layout_support_flag( $block_content, $block ) {
        $block_type            = WP_Block_Type_Registry::get_instance()->get_registered( $block['blockName'] );
        $block_supports_layout = block_has_support( $block_type, 'layout', false ) || block_has_support( $block_type, '__experimentalLayout', false );
        $child_layout          = isset( $block['attrs']['style']['layout'] ) ? $block['attrs']['style']['layout'] : null;
    
        if ( ! $block_supports_layout && ! $child_layout ) {
            return $block_content;
        }
    
        $outer_class_names = array();
    
        if ( $child_layout ) {
            $container_content_class   = wp_unique_prefixed_id( sprintf('wp-%d-container-content-', self::$block_post_id) ); // NOTE: Added post id to prefix
            $child_layout_declarations = array();
            $child_layout_styles       = array();
    
            $self_stretch = isset( $child_layout['selfStretch'] ) ? $child_layout['selfStretch'] : null;
    
            if ( 'fixed' === $self_stretch && isset( $child_layout['flexSize'] ) ) {
                $child_layout_declarations['flex-basis'] = $child_layout['flexSize'];
                $child_layout_declarations['box-sizing'] = 'border-box';
            } elseif ( 'fill' === $self_stretch ) {
                $child_layout_declarations['flex-grow'] = '1';
            }
    
            if ( isset( $child_layout['columnSpan'] ) ) {
                $column_span                              = $child_layout['columnSpan'];
                $child_layout_declarations['grid-column'] = "span $column_span";
            }
            if ( isset( $child_layout['rowSpan'] ) ) {
                $row_span                              = $child_layout['rowSpan'];
                $child_layout_declarations['grid-row'] = "span $row_span";
            }
            $child_layout_styles[] = array(
                'selector'     => ".$container_content_class",
                'declarations' => $child_layout_declarations,
            );
    
            if ( isset( $child_layout['columnSpan'] ) && ( isset( $block['parentLayout']['minimumColumnWidth'] ) || ! isset( $block['parentLayout']['columnCount'] ) ) ) {
                $column_span_number  = floatval( $child_layout['columnSpan'] );
                $parent_column_width = isset( $block['parentLayout']['minimumColumnWidth'] ) ? $block['parentLayout']['minimumColumnWidth'] : '12rem';
                $parent_column_value = floatval( $parent_column_width );
                $parent_column_unit  = explode( $parent_column_value, $parent_column_width );
    
                if ( count( $parent_column_unit ) <= 1 ) {
                    $parent_column_unit  = 'rem';
                    $parent_column_value = 12;
                } else {
                    $parent_column_unit = $parent_column_unit[1];
    
                    if ( ! in_array( $parent_column_unit, array( 'px', 'rem', 'em' ), true ) ) {
                        $parent_column_unit = 'rem';
                    }
                }
    
                $default_gap_value     = 'px' === $parent_column_unit ? 24 : 1.5;
                $container_query_value = $column_span_number * $parent_column_value + ( $column_span_number - 1 ) * $default_gap_value;
                $container_query_value = $container_query_value . $parent_column_unit;
    
                $child_layout_styles[] = array(
                    'rules_group'  => "@container (max-width: $container_query_value )",
                    'selector'     => ".$container_content_class",
                    'declarations' => array(
                        'grid-column' => '1/-1',
                    ),
                );
            }
    
            $child_css = wp_style_engine_get_stylesheet_from_css_rules(
                $child_layout_styles,
                array(
                    'context'  => 'block-supports',
                    'prettify' => false,
                )
            );
    
            if ( $child_css ) {
                $outer_class_names[] = $container_content_class;
            }
        }
    
        $processor = new WP_HTML_Tag_Processor( $block_content );
    
        if ( ! $processor->next_tag() ) {
            return $block_content;
        }
    
        if ( ! $block_supports_layout && ! empty( $outer_class_names ) ) {
            foreach ( $outer_class_names as $class_name ) {
                $processor->add_class( $class_name );
            }
            return $processor->get_updated_html();
        } elseif ( ! $block_supports_layout ) {
            return $block_content;
        }
    
        $global_settings = wp_get_global_settings();
        $fallback_layout = isset( $block_type->supports['layout']['default'] )
            ? $block_type->supports['layout']['default']
            : array();
        if ( empty( $fallback_layout ) ) {
            $fallback_layout = isset( $block_type->supports['__experimentalLayout']['default'] )
                ? $block_type->supports['__experimentalLayout']['default']
                : array();
        }
        $used_layout = isset( $block['attrs']['layout'] ) ? $block['attrs']['layout'] : $fallback_layout;
    
        $class_names        = array();
        $layout_definitions = wp_get_layout_definitions();

        // NOTE: Added post_id to prefix and changed to sprintf
        $container_class = wp_unique_prefixed_id(
            sprintf(
                'wp-%d-container-%s-is-layout-',
                self::$block_post_id,
                sanitize_title( $block['blockName'] )
            )
        );
    
        if ( isset( $used_layout['inherit'] ) && $used_layout['inherit'] || isset( $used_layout['contentSize'] ) && $used_layout['contentSize'] ) {
            $used_layout['type'] = 'constrained';
        }
    
        $root_padding_aware_alignments = isset( $global_settings['useRootPaddingAwareAlignments'] )
            ? $global_settings['useRootPaddingAwareAlignments']
            : false;
    
        if (
            $root_padding_aware_alignments &&
            isset( $used_layout['type'] ) &&
            'constrained' === $used_layout['type']
        ) {
            $class_names[] = 'has-global-padding';
        }
    
        if ( ! empty( $block['attrs']['layout']['orientation'] ) ) {
            $class_names[] = 'is-' . sanitize_title( $block['attrs']['layout']['orientation'] );
        }
    
        if ( ! empty( $block['attrs']['layout']['justifyContent'] ) ) {
            $class_names[] = 'is-content-justification-' . sanitize_title( $block['attrs']['layout']['justifyContent'] );
        }
    
        if ( ! empty( $block['attrs']['layout']['flexWrap'] ) && 'nowrap' === $block['attrs']['layout']['flexWrap'] ) {
            $class_names[] = 'is-nowrap';
        }
    
        if ( isset( $used_layout['type'] ) ) {
            $layout_classname = isset( $layout_definitions[ $used_layout['type'] ]['className'] )
                ? $layout_definitions[ $used_layout['type'] ]['className']
                : '';
        } else {
            $layout_classname = isset( $layout_definitions['default']['className'] )
                ? $layout_definitions['default']['className']
                : '';
        }
    
        if ( $layout_classname && is_string( $layout_classname ) ) {
            $class_names[] = sanitize_title( $layout_classname );
        }
    
        if ( ! current_theme_supports( 'disable-layout-styles' ) ) {
    
            $gap_value = isset( $block['attrs']['style']['spacing']['blockGap'] )
                ? $block['attrs']['style']['spacing']['blockGap']
                : null;
            if ( is_array( $gap_value ) ) {
                foreach ( $gap_value as $key => $value ) {
                    $gap_value[ $key ] = $value && preg_match( '%[\\\(&=}]|/\*%', $value ) ? null : $value;
                }
            } else {
                $gap_value = $gap_value && preg_match( '%[\\\(&=}]|/\*%', $gap_value ) ? null : $gap_value;
            }
    
            $fallback_gap_value = isset( $block_type->supports['spacing']['blockGap']['__experimentalDefault'] )
                ? $block_type->supports['spacing']['blockGap']['__experimentalDefault']
                : '0.5em';
            $block_spacing      = isset( $block['attrs']['style']['spacing'] )
                ? $block['attrs']['style']['spacing']
                : null;
    
            $should_skip_gap_serialization = wp_should_skip_block_supports_serialization( $block_type, 'spacing', 'blockGap' );
    
            $block_gap             = isset( $global_settings['spacing']['blockGap'] )
                ? $global_settings['spacing']['blockGap']
                : null;
            $has_block_gap_support = isset( $block_gap );
    
            $style = wp_get_layout_style(
                ".$container_class",
                $used_layout,
                $has_block_gap_support,
                $gap_value,
                $should_skip_gap_serialization,
                $fallback_gap_value,
                $block_spacing
            );
    
            if ( ! empty( $style ) ) {
                $class_names[] = $container_class;
                self::$block_styles .= $style; // NOTE: Appending block_styles
            }
        }
    
        $block_name    = explode( '/', $block['blockName'] );
        $class_names[] = 'wp-block-' . end( $block_name ) . '-' . $layout_classname;
    
        if ( ! empty( $outer_class_names ) ) {
            foreach ( $outer_class_names as $outer_class_name ) {
                $processor->add_class( $outer_class_name );
            }
        }
    
        $inner_block_wrapper_classes = null;
        $first_chunk                 = isset( $block['innerContent'][0] ) ? $block['innerContent'][0] : null;
        if ( is_string( $first_chunk ) && count( $block['innerContent'] ) > 1 ) {
            $first_chunk_processor = new WP_HTML_Tag_Processor( $first_chunk );
            while ( $first_chunk_processor->next_tag() ) {
                $class_attribute = $first_chunk_processor->get_attribute( 'class' );
                if ( is_string( $class_attribute ) && ! empty( $class_attribute ) ) {
                    $inner_block_wrapper_classes = $class_attribute;
                }
            }
        }
    
        do {
            if ( ! $inner_block_wrapper_classes ) {
                break;
            }
    
            $class_attribute = $processor->get_attribute( 'class' );
            if ( is_string( $class_attribute ) && str_contains( $class_attribute, $inner_block_wrapper_classes ) ) {
                break;
            }
        } while ( $processor->next_tag() );
    
        foreach ( $class_names as $class_name ) {
            $processor->add_class( $class_name );
        }
    
        return $processor->get_updated_html();
    }

}
