/*
Plugin Name: QLess Wordpress Integration
Plugin URI: http://cleverfile.net/plugins/qless/
Description: This plugin is designed to integrate with the QLess API to provide widgets and shortcodes to your site. It is developed by Clever Ogre in Pensacola, Florida.
Version: 0.2
Author: CleverOgre
Author URI: http://cleverogre.com/
Text Domain: qless
License: GPLv2 or later
License URI: http://www.gnu.org/licenses/gpl-2.0.html
Copyright: CleverOgre, Inc.
           ___
          (   )
  .--.     | |    .--.       .--.        .--.
 /    \    | |   /    \    /  _  \     /  _  \
|  .-. '   | |  |  .-. ;  . .' `. ;   . .' `. ;
| |  | |   | |  |  | | |  | '   | |   | '   | |
| |  | |   | |  |  |/  |  _\_`.(___)  _\_`.(___)
| |  | |   | |  |  ' _.' (   ). '.   (   ). '.
| '  | |   | |  |  .'.-.  | |  `\ |   | |  `\ |
' `-'  |   | |  '  `-' /  ; '._,' '   ; '._,' '
 `._ / |  (___)  `.__.'    '.___.'     '.___.'
     | |
    (___)
*/

// Typings
/// <reference path="./typings/jquery.d.ts"/>
/// <reference path="./typings/qless.d.ts"/>

// Helper Classes
/// <reference path="./helpers/helper.ts"/>
/// <reference path="./helpers/cookie.ts"/>

class QLessAppointments {
    private fadeDuration:number = 250;
    private fadeOffset:number = 125;

    private form:JQuery;
    private results:JQuery;

    private busy:boolean;

    private cookie_authentication:Cookie;
    private cookie_identity:Cookie;

    constructor($:JQueryStatic = jQuery) {
        // Grab it by the form!
        this.form = $('form.qless-form.qless-form-appointments');
        if (!Helper.validElement(this.form)) return;

        this.results = $('ul.qless-results');
        if (!Helper.validElement(this.form)) return;

        // Get Authentication Cookies
        this.cookie_authentication = new Cookie('au');
        this.cookie_identity = new Cookie('i');

        // Clear Authentication
        this.cookie_authentication.remove();
        this.cookie_identity.remove();

        // Dynamic Input Selector
        this.form.find('input[type="radio"][name="qless_input"]').on('click', () => this.updateInputSelector($));
        this.updateInputSelector($);

        // Handle Ajax, bro!
        this.form.find('[type="submit"]').on('click', (e) => this.submitForm(e, $));
    }

    // Form Functions

    private updateInputSelector($:JQueryStatic = jQuery):boolean {
        var selected = this.form.find('input[type="radio"][name="qless_input"]:checked, input[type="hidden"][name="qless_input"]');
        if (!Helper.validElement(selected)) return false;

        var inputOptions:string|Array<string> = selected.attr('data-input'); // selected.val();
        if (!Helper.validString(inputOptions)) return false;
        inputOptions = inputOptions.split(',');

        var inputs = $();
        for (var i = 0; i < inputOptions.length; i++) {
            var input = this.form.find('input[name="' + inputOptions[i] + '"]');
            if (!Helper.validElement(input)) continue;
            inputs = inputs.add(input);
        }
        if (!Helper.validElement) return false;

        return this.setInputs(inputs);
    }

    private setInputs(inputs:JQuery, $:JQueryStatic = jQuery):boolean {
        this.form.find('ul.qless-fields > li.qless-field').removeClass('qless-field-hidden').filter(function (i, li) {
            return !$(li).hasClass('qless-field-input') && !$(li).is(inputs.closest('li.qless-field'));
        }).addClass('qless-field-hidden');

        return true;
    }

    private submitForm(e:JQueryEventObject, $:JQueryStatic = jQuery) {
        var __this:QLessAppointments = this;
        e.preventDefault();

        if (this.busy == true) return;
        this.startAjax();

        this.authenticate($)
        .then(() => __this.getAppointments($))
        .then((data) => __this.buildAppointments(data, $))
        .then(function () {
            __this.stopAjax();
        }).fail(function () {
            console.log('Unable to search');
            __this.stopAjax();
        });
    }

    // Ajax Functions

    private startAjax() {
        this.busy = true;
        this.results.addClass('qless-results-ajax');

        // Remove that darn old data
        this.results.empty();
    }

    private stopAjax() {
        this.results.removeClass('qless-results-ajax');
        this.busy = false;
    }

    private buildAjaxData(action:string):object {
        // Build data
        var formData = this.form.serializeArray();
        var ajaxData = {
            'action': action,
        };
        for (var i = 0; i < formData.length; i++) {
            ajaxData[formData[i].name] = formData[i].value;
        }
        return ajaxData;
    }

    private getHost(uri:string):string {
        var parser = document.createElement('a');
        parser.href = uri;
        return parser.hostname;
    }

    private authenticate($:JQueryStatic = jQuery):JQueryPromise<boolean> {
        var __this:QLessAppointments = this;
        var d:JQueryDeferred<boolean> = jQuery.Deferred();

        if (Helper.validString(this.cookie_authentication.read()) && Helper.validString(this.cookie_identity.read())) {
            setTimeout(function () { d.resolve(); }, 1);
            return d.promise();
        }

        this.ajaxRequest('authenticate', false, 'response', $).done(function (response) {
            d.resolve(true);
        }).fail(function () {
            d.resolve(false);
        });

        return d.promise();
    }

    private wssid($:JQueryStatic = jQuery) {
        var __this:QLessAppointments = this;
        var d:JQueryDeferred<object|string|boolean> = jQuery.Deferred();

        this.qlessRequest('api/v1/wssid', false, false, 'xml', $).done(function (wssid) {
            if (Helper.validObject(wssid)) { // (Helper.validString(wssid)) {
                d.resolve(wssid);
            } else {
                d.resolve(false);
            }
        });

        return d.promise();
    }

    private getAppointments($:JQueryStatic = jQuery):JQueryPromise<boolean|object> {
        var __this:QLessAppointments = this;
        var d:JQueryDeferred<boolean|object> = jQuery.Deferred();

        this.ajaxRequest('get_appointments', false, 'json', $).done(function (data:object|boolean) {
            d.resolve(data);
        }).fail(function () {
            d.resolve(false);
        });

        return d.promise();
    }

    private buildAppointments(data, $:JQueryStatic = jQuery):JQueryPromise<boolean> {
        var __this:QLessAppointments = this;
        var d:JQueryDeferred<boolean> = jQuery.Deferred();

        this.ajaxRequest('build_appointments', { 'appointments': data }, 'html', $)
        .then(function (data) {
            var items = $(data);
            var $items = $();
            items.each(function (i) {
                if (this.nodeType != 3) {
                    $items = $items.add($(this));
                }
            });
            return $items;
        }).then((items) => __this.addResults(items, $))
        .then(function (success) {
            d.resolve(success);
        }).fail(function () {
            console.log('Unable to build results.');
            d.resolve(false);
        });

        return d.promise();
    }

    private buildError(message, $:JQueryStatic = jQuery):JQueryPromise<boolean> {
        var __this:QLessAppointments = this;
        var d:JQueryDeferred<boolean> = jQuery.Deferred();

        this.ajaxRequest('build_error', { 'message': message }, 'html', $)
        .then(function (data) {
            var items = $(data);
            var $items = $();
            items.each(function (i) {
                if (this.nodeType != 3) {
                    $items = $items.add($(this));
                }
            });
            return $items;
        }).then((items) => __this.addResults(items, $))
        .then(function (success) {
            d.resolve(success);
        }).fail(function () {
            console.log('Unable to create error message result.');
            d.resolve(false);
        });

        return d.promise();
    }

    private addResults(items:JQuery, $:JQueryStatic = jQuery):JQueryPromise<boolean> {
        var __this:QLessAppointments = this;
        var d:JQueryDeferred<boolean> = $.Deferred();

        items.css({
            'opacity': 0,
            'visibility': 'hidden',
        });
        this.results.append(items);

        items.each(function (i) {
            $(this).css('visibility', 'visible').animate({
                'opacity': 1,
            }, __this.fadeDuration + i * __this.fadeOffset, 'swing', function () {
                if (i >= items.length - 1) d.resolve(true);
            });
        });

        return d.promise();
    }

    private request(uri:string, data:object|boolean, post:boolean = false, type:string = 'json', $:JQueryStatic = jQuery):JQueryPromise<string|object|boolean> {
        var __this:QLessAppointments = this;
        var d:JQueryDeferred<string|object|boolean> = jQuery.Deferred();

        var settings:any = {
            mode: 'cors',
            cache: 'no-cache',
            credentials: 'include', // Add cookies
            headers: new Headers({
                'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8',
                'Host': this.getHost(uri),
                'Accept': '*/*',
            }),
            redirect: 'manual',
            referrer: 'client',
        };

        if (post == true) {
            settings.method = 'POST';
            settings.body = $.param(data);
            settings.headers.append('Content-Length', settings.body.length.toString());
        } else {
            settings.method = 'GET';
            uri += '?' + $.param(data);
        }

        switch (type) {
            case 'json':
                fetch(uri, settings)
                .then(response => response.text())
                .then(text => $.parseJSON(text))
                .then(function(data) {
                    d.resolve(data);
                }).catch(function (error) {
                    console.log('Request failed', error);
                    d.resolve(false);
                });
                break;
            case 'xml':
                fetch(uri, settings)
                .then(response => response.text())
                .then(text => $.parseXML(text))
                .then(function(data) {
                    d.resolve(data);
                }).catch(function (error) {
                    console.log('Request failed', error);
                    d.resolve(false);
                });
                break;
            case 'text':
            case 'html':
                fetch(uri, settings)
                .then(response => response.text())
                .then(function(text) {
                    d.resolve(text);
                }).catch(function (error) {
                    console.log('Request failed', error);
                    d.resolve(false);
                });
                break;
            default:
                fetch(uri, settings)
                .then(function(response) {
                    d.resolve(response);
                }).catch(function (error) {
                    console.log('Request failed', error);
                    d.resolve(false);
                });
                break;
        }

        return d.promise();
    }

    private ajaxRequest(action:string, data:object|boolean = false, type:string = 'json', $:JQueryStatic = jQuery):JQueryPromise<string|object|boolean> {
        var requestData = this.buildAjaxData('qless/' + action);
        if (Helper.validObject(data) && typeof data !== 'boolean') {
            for (var key in data) {
                requestData[key] = data[key];
            }
        }
        return this.request(qless.ajaxurl, requestData, true, type, $);
    }

    private qlessRequest(endpoint:string, data:object|boolean, post:boolean = false, type:string = 'xml', $:JQueryStatic = jQuery):JQueryPromise<string|object|boolean> {
        return this.request(qless.api_uri + '/' + endpoint, data, post, type, $);
    }

}

(function ($) {
    $(document).ready(function () {
        new QLessAppointments($);
    });
})(jQuery);
