<?php
/**
 * @package CleverOgre
 * @subpackage BisTrack
 * @version 0.1.0
 * @since 0.1.0
 */

namespace LearnDash\Certificate\PhpWord;

defined('ABSPATH') || exit;

use \OgreCore\Field;
use \OgreCore\Field\CheckBox as CheckBoxField;

class SettingsGroup {
    protected string $id = '';
    protected string $title = '';
    protected string $description = '';
    protected array $fields = [];

    function __construct(string $id, string $title = '', string $description = '', array $fields = []) {
        $this->id = $id;
        $this->title = $title;
        $this->description = $description;
        if (is_array($fields) && !empty($fields)) $this->add_fields($fields);
    }

    public function add_field($field):bool {
        if (!is_a($field, '\OgreCore\Field')) return false;
        $field->set_group($this->id);
        $this->fields[] = $field;
        return true;
    }
    public function add_fields(array $fields):bool {
        $result = true;
        foreach ($fields as $field) {
            if (!$this->add_field($field)) $result = false;
        }
        return $result;
    }

    public function get_id():string {
        return $this->id;
    }
    public function get_title():string {
        return $this->title;
    }
    public function has_title():bool {
        return !empty($this->title);
    }
    public function get_description():string {
        return $this->description;
    }
    public function has_description():bool {
        return !empty($this->description);
    }
    public function get_fields():array {
        return $this->fields;
    }
    public function set_field_object_type($value) {
        foreach ($this->fields as $field) {
            $field->set_object_type($value);
        }
    }

    public function get_data_value(array $data, object $field, bool $format = false) {
        $value = null;
        switch ($field->get_type()) {
            case 'checkbox':
                $value = array_key_exists($field->get_name(), $data) && $data[$field->get_name()] === 'yes';
                if (!!$format && array_key_exists($field->get_name(), $data)) $value = !!$value ? 'yes' : 'no';
                break;
            case 'number':
                $value = array_key_exists($field->get_name(), $data) && is_numeric($data[$field->get_name()]) ? wc_clean(wp_unslash($data[$field->get_name()])) : null;
                break;
            case 'text':
            default:
                $value = array_key_exists($field->get_name(), $data) ? wc_clean(wp_unslash($data[$field->get_name()])) : null;
                break;
        }
        return !is_null($value) ? $value : false;
    }

    public function save_field(int $object_id, array $data, object $field):bool {
        $value = $this->get_data_value($data, $field, false);
        if (!is_null($value)) return $field->set($value, $object_id) !== false;
        return false;
    }

    public function save_fields(int $object_id, array $data):bool {
        $result = true;
        foreach ($this->fields as $field) {
            if (!$this->save_field($object_id, $data, $field)) $result = false;
        }
        return $result;
    }

    public function render_fields($object_id = false) {
        foreach ($this->fields as $field) {
            $field->build($object_id);
        }
    }
}

abstract class ObjectSettings {
    use Singleton;

    protected string $id = '';
    protected string $title = '';
    protected string $description = '';
    protected array $groups = [];

    function __construct(string $id, string $title, string $description = '') {
        $this->id = $id;
        $this->title = $title;
        $this->description = $description;

        add_action('after_setup_theme', [$this, 'init'], 10);
    }

    public function init():bool {
        // Ensure that OgreCore Field package has been included.
        if (!class_exists('\OgreCore\Field')) return false;

        $this->groups = $this->register_groups();
        if (empty($this->groups)) return false;

        return true;
    }

    abstract protected function register_groups():array;

    protected function get_nonce_action($object_id = false):string {
        if (!!$object_id) {
            return sprintf('%s_%s', $this->id, sanitize_title($object_id));
        }
        return $this->id;
    }
    protected function get_nonce_name():string {
        return sprintf('_%snonce', preg_replace('/[^a-z]/', '', $this->id));
    }
    protected function the_nonce_field($object_id = false) {
        wp_nonce_field($this->get_nonce_action($object_id), $this->get_nonce_name(), false, true);
    }
    protected function verify_nonce(array $data, $object_id = false) {
        return isset($data[$this->get_nonce_name()]) && wp_verify_nonce($data[$this->get_nonce_name()], $this->get_nonce_action($object_id));
    }

    public function render($object_id = false) {
        if (empty($this->groups)) return;

        $this->before_render($object_id);
        $this->render_groups($object_id);
        $this->after_render($object_id);
    }

    protected function before_render($object_id = false) {
        $this->the_nonce_field($object_id);
    }
    protected function render_groups($object_id = false):bool {
        if (empty($this->groups)) return false;

        foreach ($this->groups as $group) {
            if ($group->has_title()) printf('<h3>%s</h3>', esc_html($group->get_title()));
            if ($group->has_description()) {
                printf('<p class="description" style="margin-top:1em;font-style:italic;">%s</p>', esc_html($group->get_description()));
            }
            $group->render_fields($object_id);
        }

        return true;
    }
    protected function after_render($object_id = false) { }

    protected function sanitize_object_id($object_id = false) {
        if (is_a($object_id, '\WC_Product')) $object_id = $object_id->get_id();
        if (is_a($object_id, '\WP_Post')) $object_id = $object_id->ID;
        if ($object_id === false) $object_id = get_the_ID();
        if (!is_int($object_id)) return false;
        return $object_id;
    }
    protected function validate_object_id($object_id = false) {
        return !!$this->sanitize_object_id($object_id);
    }

    public function save_fields($object_id = false):bool {
        if (!$this->validate_object_id($object_id)) return false;
        $object_id = $this->sanitize_object_id($object_id);

        if (!$this->verify_nonce($_POST, $object_id)) return false;

        foreach ($this->groups as $group) {
            $group->save_fields($object_id, $_POST);
        }

        return true;
    }

    public function get_field_value(string $group_id, string $field_name, $object_id = false) {
        if (!$this->validate_object_id($object_id)) return false;
        $object_id = $this->sanitize_object_id($object_id);

        foreach ($this->groups as $group) {
            if ($group->get_id() !== $group_id) continue;
            foreach ($group->get_fields() as $field) {
                if ($field->get_name() !== $field_name) continue;
                return (string)$field->get($object_id);
            }
        }
        return false;
    }

}

abstract class PostSettings extends ObjectSettings {

    protected string $post_type = '';

    function __construct(string $post_type, string $id, string $title, string $description = '') {
        $this->post_type = $post_type;
        parent::__construct($id, $title, $description);
    }

    public function init():bool {
        if (!parent::init()) return false;

        foreach ($this->groups as $group) {
            $group->set_field_object_type(Field::ObjectTypePost);
        }

        add_action('add_meta_boxes', [$this, 'add_meta_box'], 10, 2);
        add_action('save_post', [$this, 'save_fields'], 10, 1);

        return true;
    }

    public function add_meta_box($post_type, $post) {
        add_meta_box(
            $this->id,
            $this->title,
            [$this, 'render_meta_box'],
            $this->post_type
        );
    }

    public function render_meta_box($post) {
        if ($post->post_type !== $this->post_type) return;
        $this->render($post->ID);
    }

    protected function render_groups($object_id = false):bool {
        if (empty($this->groups)) return false;

        if (count($this->groups) > 1) {
            printf(
                '<div id="%s-navigation"><h2 class="nav-tab-wrapper current" style="padding-bottom:0;padding-left:0;padding-right:0;">',
                $this->id
            );
            $i = 0;
            foreach ($this->groups as $group) {
                printf(
                    '<a class="nav-tab %s" href="javascript:;" data-tab="%s-meta-options">%s</a>',
                    (0 == $i++ ? 'nav-tab-active' : ''),
                    $group->get_id(),
                    $group->get_title()
                );
            }
            echo '</h2></div>';
        }

        $i = 0;
        foreach ($this->groups as $group) {
            printf(
                '<div id="%1$s-meta-options" class="meta-options %1$s-meta-options" style="%2$s">',
                $group->get_id(),
                (0 == $i++ ? '' : 'display:none;')
            );
            if ($group->has_description()) {
                printf('<p class="description" style="margin-top:1em;font-style:italic;">%s</p>', esc_html($group->get_description()));
            }
            $group->render_fields($object_id);
            echo '</div>';
        }

        return true;
    }
    protected function after_render($object_id = false) {
        parent::after_render($object_id);
?>
<script type="text/javascript">
(function ($) {
    $('#<?php echo $this->id; ?>-navigation a').on('click', function (e) {
        e.preventDefault();
        $(this).closest('#<?php echo $this->id; ?>').parent().find('.meta-options').hide();
        $('#' + $(this).attr('data-tab')).show();
        $('#<?php echo $this->id; ?>-navigation a').removeClass('nav-tab-active');
        $(this).addClass('nav-tab-active');
    });
})(jQuery);
</script>
<?php
    }

    protected function validate_object_id($object_id = false) {
        if (!parent::validate_object_id($object_id)) return false;
        return get_post_type(parent::sanitize_object_id($object_id)) === $this->post_type;
    }

}
