<?php
/**
 * @package CleverOgre
 * @subpackage GFMatadorJobs
 * @since 0.1.0
 * @version 0.1.0
 */

namespace GFMatadorJobs;

use \matador\Matador;
use \matador\MatadorJobs\Email\AdminNoticeGeneralMessage;

if (!defined('ABSPATH')) exit;

class Feed extends \GFFeedAddOn {

    protected $_version = GFMJ_ADDON_VERSION;
    protected $_min_gravityforms_version = '2.5';
    protected $_slug = 'gfmatadorjobs';
    protected $_path = 'gravityforms-matador-jobs/gravityforms-matador-jobs.php';
    protected $_full_path = __FILE__;
    protected $_title = 'Matador Jobs Add-On';
    protected $_short_title = 'Matador Jobs';

    private static $_instance = null;

    public static function get_instance() {
        if (self::$_instance == null) self::$_instance = new Feed();
        return self::$_instance;
    }

    public function init() {
        parent::init();
    }

    public function feed_settings_fields() {
        $path = Plugin::get_path() . 'assets/json/feed.json';
        if (!file_exists($path)) return [];
        $settings = json_decode(file_get_contents($path), true);
        if (is_null($settings)) return [];
        // Field mappings gathered from \matador\Application_Handler::application_fields_structure();
        return $settings;
    }

    public function feed_list_columns() {
        return [
            'feedName' => __('Name'),
        ];
    }

    public function get_menu_icon() {
        return 'dashicons-businessman';
    }

    public function process_feed($feed, $entry, $form) {
        $this->log_debug(__METHOD__ . '(): Start feed processing');

        $target = self::get_feed_target($feed);
        $this->log_debug(__METHOD__ . '(): Provided Matador target: ' . $target);

        $map = self::get_dynamic_field_map_fields($feed, 'application_fields');
        if (empty($map)) {
            $this->log_debug(__METHOD__ . '(): No field mappings provided.');
            return false;
        }

        $data = [];
        foreach ($map as $target_field_name => $source_field_id) {
            $is_subfield = strpos($source_field_id, '.') !== false;
            $field = self::get_form_field_by_id($form, $source_field_id);

            if ($field === null) {
                $this->log_debug(__METHOD__ . sprintf('(): GF field with id "%s" wasn\'t found', $source_field_id));
                continue;
            }

            $data[$target_field_name] = rgar($entry, $source_field_id);
        }
        if (empty(array_filter($data))) {
            $this->log_debug(__METHOD__ . '(): No valid field mappings.');
            return false;
        }

        $processed_data = $this->process_application($data, $entry);
        if (!$process_data) {
            $this->log_debug(__METHOD__ . '(): Unable to process application.');
            $this->add_note(rgar($entry, 'id'), __('Unable to process entry into Matador Jobs application data.', 'gravityforms-matador-jobs'), 'error');
            return false;
        }

        $application = apply_filters('matador_application_data_processed', $processed_data, $this->request);

        $application_id = $this->save_application($application);
        if (!is_wp_error($application_id) && is_int($application_id) && $application_id > 0) {
            $this->add_note(rgar($entry, 'id'), sprintf(
                __('Successfully recorded entry as Matador Jobs Application. <a href="%s">View Application</a>', 'gravityforms-matador-jobs'),
                get_edit_post_link($application_id)
            ), 'success');
            return $application_id;
        } else {
            $this->add_note(rgar($entry, 'id'), __('Failed to save entry data as Matador Jobs application post.', 'gravityforms-matador-jobs'), 'error');
            return false;
        }
    }

    protected function process_application($fields, $entry) {
        // Based on \matador\Application_Handler->process_application()
        if (empty(array_filter($fields))) return false;

        self::$used = [];
        $ignored = apply_filters('matador_application_handler_start_ignored_fields', ['type']);
        self::add_used_fields($ignored);

        $application = [
            'type' => 'application',
            'application' => [
                'ip' => rgar($entry, 'ip'),
            ],
            'post_content' => '',
        ];

        $application = self::get_name($application, $fields);
        $application = self::get_bullhorn_fields($application, $fields);

        $application = apply_filters('matador_application_data_processing', $application, $submitted, $this);

        $application = self::get_message($application, $fields);
        $application = self::get_files($application, $fields);

        return $application;
    }

    protected function save_application($application) {
        $insert_post_args = apply_filters('matador_application_post_args', [
			'post_title'   => self::get_application_post_title($application),
			'post_content' => $application['post_content'],
			'post_type'    => Matador::variable('post_type_key_application'),
			'post_author'  => 1,
			'post_status'  => 'publish',
			'meta_input'   => [
				Matador::variable('application_data') => $application['application'],
				Matador::variable('candidate_sync_status') => -1,
				Matador::variable('submission_type') => ! empty($application['type']) ? $application['type'] : 'application',
			],
        ], $application, []); // Create post object

		$wp_id = wp_insert_post($insert_post_args);

		if (!is_wp_error($wp_id) || 0 !== $wp_id) {
			do_action('matador_new_job_application', $wp_id, $application['application']);
		} else {
            $this->log_debug(__METHOD__ . '(): The save of a new application failed. The data is: ' . print_r($applicaiton, true));
			do_action('matador_new_job_application_failed', $insert_post_args);

			AdminNoticeGeneralMessage::message(esc_html__('A candidate application could not be saved to WordPress for an unknown reason. The data is: ', 'matador-jobs') . print_r($application, true));
		}

		return $wp_id;
    }
    private static function get_application_post_title($application) {
		if (!empty($application)) {
			$title = [];

			if (!empty($application['application']['name']['fullName'])) {
				$title[] = $application['application']['name']['fullName'];
			} else {
				$title[] = esc_html__('An applicant', 'matador-jobs');
			}

			if (isset($application['type']) && 'referral' === $application['type']) {
				$title[] = esc_html__('was referred to', 'matador-jobs');
			} else {
				$title[] = esc_html__('applied to', 'matador-jobs');
			}
			if (isset($application['application']['jobs']) && !empty($application['application']['jobs'][0]['title'])) {
				$title[] = $application['application']['jobs'][0]['title'];
			} else {
				$title[] = esc_html__('a position', 'matador-jobs');
			}

			if (isset($application['application']['jobs']) && !empty($application['application']['jobs']) && is_array($application['application']['jobs']) && count($application['application']['jobs']) > 1 ) {
				// Translators: number of additional positions.
				$title[] = sprintf(esc_html__('and %1$s other position(s).', 'matador-jobs'), count($application['application']['jobs']) - 1);
			}

			return implode(' ', $title);
		} else {
			return esc_html__('Application received.', 'matador-jobs');
		}
	}

    private static function get_name($application, $fields) {
		$name = [];

		if (!empty($fields['name'])) {
			$name = $this->parse_single_name_field($fields['name']);
		} else {
            $name_fields = ['namePrefix', 'firstName', 'middleName', 'lastName', 'nameSuffix'];
            foreach ($name_fields as $name_field) {
                if (isset($fields[$name_field]) && !empty($fields[$name_field])) {
                    $name[$name_field] = sanitize_text_field($fields[$name_field]);
                }
            }
			if (!empty($name)) {
				$name['fullName'] = trim(implode(' ', array_filter($name)));
			}
		}

		if (!empty($name)) {
			$application['application']['name'] = $name;
			$application['post_content'] = _x('Name', 'matador-jobs', 'Label for a person\'s name. ie: "Name: John Doe"') . ': ' . $name['fullName'] . PHP_EOL . PHP_EOL;
		}

        self::add_used_fields('name', 'firstName', 'middleName', 'lastName', 'namePrefix', 'nameSuffix');

		return $application;
    }

    private static function get_bullhorn_fields($application, $fields) {
        $bullhorn_fields = apply_filters('matador-fields-to-add-to-application', [
			'email',
			'mobile',
			'phone',
			'address1',
			'address2',
			'city',
			'state',
			'zip',
			'occupation',
			'company',
            'profile'
		]);
        $bullhorn_fields[] = 'jobs';

        foreach ($fields as $field => $value) {
            if (!in_array($field, $bullhorn_fields)) continue;
            $post_text = false;
            switch ($field) {
                case 'jobs':
                    $post_ids = [];
                    if (is_numeric($value)) {
                        $post_ids[] = intval($value);
                    } else if (is_string($value)) {
                        $post_ids = json_decode($value, true);
                        if (is_null($post_ids)) $post_ids = [];
                    }
                    if (is_array($post_ids) && !empty($post_ids)) {
                        foreach ($post_ids as $post_id) {
                            if (!is_numeric($post_id)) continue;
                            $job = self::get_job_info($post_id);
                            if (!is_array($job)) continue;
                            if (!is_array($application['application']['jobs'])) $application['application']['jobs'] = [];
                            $application['application']['jobs'][] = $job;
                            $application['post_content'] .= self::get_job_info_as_string($job);
                        }
                    }
                    break;
                case 'email':
                    $application['application'][$field] = sanitize_email($value);
                    $post_text = $value;
                    break;
                default:
                    if (is_array($value)) {
                        $application['application'][$field] = array_map('sanitize_text_field', $value);
                        $post_text = implode(', ', $value);
                    } else {
                        $application['application'][$field] = sanitize_text_field($value);
                        $post_text = $value;
                    }
                    break;
            }
            if (is_string($post_text) && !empty($post_text)) {
                $application['post_content'] .= ucfirst($field) . ': ' . $post_text . PHP_EOL . PHP_EOL;
            }
        }
        self::add_used_fields($bullhorn_fields);
        self::add_used_fields('bhid', 'wpid', 'jobs', 'request');

        return $application;
    }

    private static function get_message($application, $fields) {
        $notes = [];

        if (isset($fields['message']) && !empty($fields['message'])) {
            $label = apply_filters('matador_submit_candidate_notes_message_label', __('Message: ', 'matador-jobs'));
            $notes[] = $label . esc_html(wp_strip_all_tags($fields['message']));
            self::add_used_fields('message');
        }

        foreach ($fields as $key => $field) {
            if (in_array($key, self::$used, true) || empty($field)) continue;

            $line_item = '';
            if (is_array($field)) {
                foreach ($field as $item) {
                    if (empty($item)) continue;
                    $line_item = !empty($line_item) ? $line_item . ', ' . $item : $item;
                }
            } else {
                $line_item = $field;
            }

            $line_label = ucwords($key);
            $line_label = apply_filters('matador_application_note_line_label', $line_label, $key, $field);

            $line_item = apply_filters('matador_application_note_line_item', $line_item, $key, $field);

            $notes[] = $line_label . ': ' . $line_item;
			unset($line_item);
			unset($line_label);
        }

        if (!empty($notes)) {
            $application['application']['message'] = '<p>' . implode('.</p> <p>', apply_filters('matador_application_note_content', $notes));
			$application['post_content'] .= implode(" \n\n" . PHP_EOL, $notes) . PHP_EOL;
        }

        return $application;
    }

    private static function get_files($application, $fields) {
        $file_keys = Matador::variable('application_allowed_files_keys');

        foreach ($file_keys as $key) {
            if (!isset($fields[$key]) || empty($fields[$key])) continue;

            if (!isset($application['application']['files'])) $application['application']['files'] = [];

            $files = json_decode($fields[$key], true); // multi-upload
            if ($files !== null && is_array($files) && !empty($files)) {
                foreach ($files as $url) {
                    $application = self::add_file($application, $key, $url);
                }
            } else if (wp_http_validate_url($fields[$key])) { // single upload
                $application = self::add_file($application, $key, $fields[$key]);
            }
        }

        self::add_used_fields($file_keys);

        return $application;
    }
    private static function add_file($application, $key, $url) {
        if (empty($url) || !wp_http_validate_url($url)) return $application;

        if (!isset($application['application']['files'])) $application['application']['files'] = [];

        $path = $url;
        if (strpos($path, get_site_url()) !== false) {
            $path = ABSPATH . substr($path, strlen(get_site_url()));
            $path = str_replace('//', '/', $path);
        }

        $name = basename($url);

        $application['application']['files'][$key] = [
            'url' => $url,
            'path' => $path,
            'file' => $name,
            'synced' => 0,
        ];
        $application['post_content'] .= PHP_EOL . esc_html(ucwords($key)) . ' ' . esc_html__('File', 'matador-jobs') . ': ' . sprintf('<a href="%s">%s</a>', esc_url($url), esc_html($name)) . PHP_EOL . PHP_EOL;

        return $application;
    }

    private static function parse_single_name_field($raw_name = null) {
        if (empty($raw_name) || !is_string($raw_name)) {
			return [];
		}

		$name = [];

		$raw_name = preg_replace('/\s+/', ' ', sanitize_text_field($raw_name));
		$all_parts = explode(', ', trim(esc_attr($raw_name)));
		$name_parts = explode(' ', trim($all_parts[0]));
		$suffixes = array_slice($all_parts, 1);

		$name['firstName'] = $name_parts[0];

        $allowed_suffixes = Matador::variable('application_name_suffixes');

		if (2 < count($name_parts)) {
			if (in_array(strtolower(end($name_parts)), $allowed_suffixes, true)) {
				$suffixes[] = end($name_parts);
				if (3 < count($name_parts)) $name['middleName'] = $name_parts[1];
				$name['lastName'] = $name_parts[count($name_parts) - 2];
			} else {
				$name['middleName'] = $name_parts[1];
				$name['lastName'] = end($name_parts);
			}
		} else {
			$name['lastName'] = end($name_parts);
		}

		if (!empty($suffixes)) {
			foreach ($suffixes as $suffix) {
				$name['suffix'] = isset($name['suffix']) ? $name['suffix'] . ', ' . strtoupper($suffix) : strtoupper($suffix);
			}
		}

		$name['fullName'] = trim(implode(' ', array_filter($name)));

		return $name;
    }

    private static function get_job_info($post_id) {
        if (!is_int($post_id) && is_numeric($post_id)) $post_id = intval($post_id);
        if (!is_int($post_id) || get_post_type($post_id) != 'matador-job-listings') return false;
        $_post = get_post($post_id);
        $job = [
            'title' => $_post->post_title,
            'wpid' => $_post->ID,
            'bhid' => (int)get_post_meta($_post->ID, '_matador_source_id', true),
        ];
        if (!empty($job['bhid'])) {
            $job['synced'] = 'Pending';
        } else {
            $job['synced'] = 'Not Applicable';
        }
        return $job;
    }
    private static function get_job_info_as_string($job = null) {
        if (empty($job)) return '';

        $job_title = esc_html__('Job Applied For', 'matador-jobs') . ': ' . $job['title'];

		if (!empty($job['wpid']) || !empty($job['bhid'])) {
			$job_title_suffix  = isset($job['bhid']) ? 'BHID: ' . $job['bhid'] : null;
			$job_title_suffix .= (isset($job_title_suffix) && isset($job['wpid'])) ? ' | ' : '';
			$job_title_suffix .= isset($job['wpid']) ? 'WPID: ' . $job['wpid'] : '';
			$job_title         = sprintf('%s ( %s )', $job_title, $job_title_suffix) . PHP_EOL . PHP_EOL;
		}

		return $job_title;
    }

    private static function get_feed_target($feed) {
        $target = trim(rgars($feed, 'meta/target'));
        if (empty($target)) return 'application';
        return $target;
    }

    private static function get_form_field_by_id($form, $field_id) {
        $field_id = intval($field_id);
        foreach ($form['fields'] as &$field) {
            if ($field->id == $field_id) return $field;
        }
        return null;
    }

    private static $used;
    public static function add_used_fields() {
        if (!is_array(self::$used)) self::$used = [];
        $added = func_get_args();
        foreach ($added as $field) {
            if (is_string($field)) {
                self::$used[] = $field;
            } else if (is_array($field)) {
                foreach ($field as $included) {
                    self::$used[] = $included;
                }
            }
        }
    }

}
