/*
Plugin Name: OgreModal
Plugin URI: http://cleverfile.net/plugins/ogremodal/
Description: OgreModal is a plugin developed by CleverOgre in Pensacola, Florida.
Version: 0.4.3
Since: 0.1.0
Author: CleverOgre
Author URI: http://cleverogre.com/
Text Domain: ogremodal
License: GPLv2 or later
License URI: http://www.gnu.org/licenses/gpl-2.0.html
Copyright: CleverOgre, Inc.
*/

function OgreModal() { }

// Cookie functions

OgreModal.createCookie = function (name, value, seconds) {
    var expires = "";
    if (seconds) {
        var date = new Date();
        date.setTime(date.getTime() + (seconds * 1000));
        expires = "; expires=" + date.toUTCString();
    }
    document.cookie = name + "=" + value + expires + "; path=/";
};

OgreModal.readCookie = function (name) {
    var nameEQ = name + "=";
    var ca = document.cookie.split(';');
    for (var i = 0; i < ca.length; i++) {
        var c = ca[i];
        while (c.charAt(0) == ' ') c = c.substring(1, c.length);
        if (c.indexOf(nameEQ) === 0) return c.substring(nameEQ.length, c.length);
    }
    return null;
};

OgreModal.eraseCookie = function (name) {
    OgreModal.createCookie(name, "", -1 * 24 * 60 * 60); // -1 days
};

// General Functions

OgreModal.validObject = function (obj) {
    return typeof obj !== 'undefined' && obj !== null;
};
OgreModal.validString = function (str) {
    return OgreModal.validObject(str) && typeof str.trim !== 'undefined' && str.trim().length > 0;
};
OgreModal.validArray = function (arr) {
    return OgreModal.validObject(arr) && typeof arr.length !== 'undefined' && arr.length > 0;
};
OgreModal.validElement = function (elem) {
    return OgreModal.validObject(elem) && typeof elem.length !== 'undefined' && elem.length > 0;
};

OgreModal.addClass = function (elem, name) {
    return elem.add('body').addClass(name);
};
OgreModal.removeClass = function (elem, name) {
    return elem.add('body').removeClass(name);
};

// Pageview Functions

OgreModal.trackPageviews = function () {
    var key = 'ogremodal_pageviews';
    var pageviews = 0;

    // Retrieve
    if ('sessionStorage' in window) {
        pageviews = window.sessionStorage.getItem(key);
    } else {
        pageviews = OgreModal.readCookie(key);
    }

    // Parse
    if (pageviews !== null) pageviews = parseInt(pageviews, 10);
    if (isNaN(pageviews) || pageviews === null) pageviews = 0;

    // Increment
    pageviews++;

    // Store
    if ('sessionStorage' in window) {
        window.sessionStorage.setItem(key, pageviews);
    } else {
        OgreModal.createCookie(key, pageviews, 60 * 60); // Store for 1 hour
    }

    return pageviews;
};

OgreModal.getPageviews = function () {
    var key = 'ogremodal_pageviews';
    var pageviews = 1;

    // Retrieve
    if ('sessionStorage' in window) {
        pageviews = window.sessionStorage.getItem(key);
    } else {
        pageviews = OgreModal.readCookie(key);
    }

    // Parse
    if (pageviews !== null) pageviews = parseInt(pageviews, 10);
    if (isNaN(pageviews) || pageviews === null) pageviews = 1;

    return pageviews;
};

// Modal Functions

OgreModal.validModal = function (modal, $) {
    if (!OgreModal.validElement(modal)) return false;
    if (typeof $ === 'undefined') $ = jQuery;

    // Check if bot crawler
    if (/bot|googlebot|crawler|spider|robot|crawling/i.test(navigator.userAgent)) return false;

    // Pageviews setting & cookie
    var minPageviews = modal.get(0).hasAttribute('data-pageviews') ? parseInt(modal.attr('data-pageviews'), 10) : 1;
    if (isNaN(minPageviews) || minPageviews === null) minPageviews = 1;
    var cookiePageviews = OgreModal.getPageviews();

    // Check pageviews
    if (modal.attr('data-dismiss_duration') !== 'trigger' && cookiePageviews < minPageviews) return false;

    // Dismiss time cookie (per-modal)
    var cookieTime = OgreModal.readCookie('ogremodal_' + modal.attr('data-id'));
    if (cookieTime !== null) cookieTime = parseInt(cookieTime, 10);
    if (isNaN(cookieTime) || cookieTime === null) cookieTime = false;

    if (modal.attr('data-dismiss_immediate') === '1' && cookieTime == false) {
        OgreModal.createCookie('ogremodal_' + modal.attr('data-id'), Math.floor(Date.now() / 1000), parseInt(modal.attr('data-dismiss_duration'), 10)); // Save Cookie
    }

    // Check dismiss
    if ($.inArray(modal.attr('data-dismiss_duration'), ['page', 'trigger']) == -1 && cookieTime != false && Math.floor(Date.now() / 1000) - cookieTime < parseInt(modal.attr('data-dismiss_duration'), 10)) return false;

    return true;
};

OgreModal.openModal = function (modal, useAnimation, $) {
    if (typeof useAnimation === 'undefined') useAnimation = false;
    if (typeof $ === 'undefined') $ = jQuery;
    if (!OgreModal.validElement(modal)) return false;

    OgreModal.addClass(modal, 'ogremodal-active');

    $('html, body, #page').css('overflow', 'hidden'); // Lock Scroll

    var animationComplete = function () {
        modal.show();
        OgreModal.addClass(modal, 'ogremodal-open');
    };
    if (useAnimation == true) {
        OgreModal.modalAnimation(modal, 'in', $).then(animationComplete);
    } else {
        animationComplete();
    }

    return true;
};

OgreModal.closeModal = function (modal, useAnimation, $) {
    if (typeof useAnimation === 'undefined') useAnimation = true;
    if (typeof $ === 'undefined') $ = jQuery;
    if (!OgreModal.validElement(modal)) return false;

    if (modal.attr('data-dismiss_duration') != 'page') {
        OgreModal.createCookie('ogremodal_' + modal.attr('data-id'), Math.floor(Date.now() / 1000), parseInt(modal.attr('data-dismiss_duration'), 10)); // Save Cookie
    }

    OgreModal.removeClass(modal, 'ogremodal-open');

    $('html, body, #page').css('overflow', ''); // Release Scroll

    var animationComplete = function () {
        modal.hide();
        OgreModal.removeClass(modal, 'ogremodal-active');
        if (modal.attr('data-dismiss_duration') != 'trigger') {
            OgreModal.removeModal(modal);
        }
    };

    if (useAnimation == true) {
        OgreModal.modalAnimation(modal, 'out', $).then(animationComplete);
    } else {
        animationComplete();
    }

    return true;
};

OgreModal.modalAnimation = function (modal, direction, $) {
    if (typeof direction === 'undefined') direction = 'out';
    if (typeof $ === 'undefined') $ = jQuery;

    var d = $.Deferred();

    var transition_type = modal.attr('data-transition_type');
    if (!OgreModal.validString(transition_type)) transition_type = modal.attr('data-transition_style');

    modal.addClass('ogremodal-transition');
    modal.addClass('ogremodal-transition-' + direction);
    modal.addClass('ogremodal-transition-' + transition_type);

    var transitionComplete = function () {
        modal.removeClass('ogremodal-transition');
        modal.removeClass('ogremodal-transition-' + direction);
        modal.removeClass('ogremodal-transition-' + transition_type);

        d.resolve(transition_type);
    };

    var transition_speed = modal.attr('data-transition_speed');
    if (isNaN(transition_speed)) transition_speed = '1';
    transition_speed = parseFloat(transition_speed) * 1000;

    switch (transition_type) {
        case 'slide':
            if (direction == 'in') {
                modal.slideDown(transition_speed, transitionComplete);
            } else {
                modal.slideUp(transition_speed, transitionComplete);
            }
            break;
        case 'custom':
            setTimeout(transitionComplete, transition_speed);
            break;
        case 'fade':
        default:
            if (direction == 'in') {
                modal.fadeIn(transition_speed, transitionComplete);
            } else {
                modal.fadeOut(transition_speed, transitionComplete);
            }
            break;
    }

    return d.promise();
};

OgreModal.removeModal = function (modal) {
    if (!OgreModal.validElement(modal)) return false;

    modal.remove();
};

(function ($) {
    function documentReady() {
        OgreModal.trackPageviews();

        var modals = $('article.ogremodal-modal');
        if (!OgreModal.validElement(modals)) return;

        modals.each(function () {
            var modal = $(this);
            if (!OgreModal.validModal(modal, $)) return;

            // Setup Events
            var dismissClick = function (e) {
                e.preventDefault();
                OgreModal.closeModal(modal, true, $);
                return true;
            };
            var outerClick = function (e) {
                OgreModal.closeModal(modal, true, $);
                return true;
            };
            var innerClick = function (e) {
                if (e.stopPropagation) e.stopPropagation();
                return true;
            };
            if (typeof $.fn.on !== 'undefined') {
                modal.find('a.ogremodal-dismiss, a.dismiss, .dismiss > a').on('click', dismissClick);
                if (modal.attr('data-click_out') === '1') {
                    modal.on('click', outerClick);
                    modal.find('.ogremodal-container').on('click', innerClick);
                }
            } else {
                modal.find('a.ogremodal-dismiss, a.dismiss, .dismiss > a').click(dismissClick);
                if (modal.attr('data-click_out') === '1') {
                    modal.click(outerClick);
                    modal.find('.ogremodal-container').click(innerClick);
                }
            }

            if (modal.attr('data-dismiss_duration') == 'trigger') {
                var triggerClick = function (e) {
                    e.preventDefault();
                    OgreModal.openModal(modal, true, $);
                    return true;
                };
                var trigger = $('[href="#' + modal.attr('data-id') + '"]');
                if (!OgreModal.validElement(trigger)) {
                    trigger = $('a.ogremodal-trigger, .ogremodal-trigger a');
                }
                if (typeof $.fn.on !== 'undefined') {
                    trigger.on('click', triggerClick);
                } else {
                    trigger.click(triggerClick);
                }
            } else if (modal.get(0).hasAttribute('data-delay') && typeof modal.data('delay') === 'number' && modal.data('delay') > 0) {
                setTimeout(function () {
                    OgreModal.openModal(modal, true, $);
                }, modal.data('delay') * 1000);
            } else {
                // Open Modal
                OgreModal.openModal(modal, false, $);
            }
        });
    }

    $(document).ready(documentReady);
})(jQuery);
