<?php
/**
 * @package CleverOgre
 * @subpackage WP Job Manager - JobAdder Integration
 * @since WP Job Manager - JobAdder Integration 0.1.1
 */

namespace JobAdder;

if (!defined('ABSPATH')) exit;

class Settings {

    private static $defaults;

    // Plugin Settings
    public static $name;
    public static $version;
    public static $capability;
    public static $path;
    public static $dir;
    public static $hook;

    // API Settings
    public static $client_id;
    public static $client_secret;
    public static $authorized_ips;
    public static $auth_code;
    public static $access_token;
    public static $token_expires;
    public static $refresh_token;
    public static $api_url;
    public static $webhook_id;

    // Debug Settings
    public static $debug_enabled;
    public static $debug_email;

    public static function init() {
        self::$defaults = array(
            'authorized_ips' => '',
            'auth_code' => '',
            'access_token' => '',
            'token_expires' => '0',
            'refresh_token' => '',
            'api_url' => 'https://api.jobadder.com/v2/',
            'webhook_id' => '',
            'debug_enabled' => '0',
            'debug_email' => '',
        );

        $options = get_option('wpjm-jobadder_options');
        if (!is_array($options)) $options = array();
    }

    public static function get($key) {
        if (!property_exists(__CLASS__, $key)) return false;

        switch ($key) {
            default:
                $options = get_option('wpjm-jobadder_options');

                if (isset($options["settings_{$key}"])) {
                    return $options["settings_{$key}"];
                } else if (!isset(self::$defaults[$key])) {
                    return self::$defaults[$key];
                }

                return false;
        }


        return null;
    }

    public static function set($key, $value) {
        if (!property_exists(__CLASS__, $key)) return false;

        switch ($key) {
            /* // Read-only properties
            case 'readonly':
                break;
            */

            default:
                $options = get_option('wpjm-jobadder_options');
                $options["settings_{$key}"] = $value;
                update_option('wpjm-jobadder_options', $options);
                break;
        }

        return true;
    }

    public static function get_default($key) {
        if (!property_exists(__CLASS__, $key)) return null;

        if (!array_key_exists($key, self::$defaults)) return null;

        return self::$defaults[$key];
    }

}

class SettingsPage {

    public static function init() {
        add_action('admin_init', array(__CLASS__, 'register_settings'));
        add_action('admin_menu', array(__CLASS__, 'menu'));
    }

    static function menu() {
        if (!current_user_can(\JobAdder\Settings::get('capability'))) return;

        $page = add_submenu_page('edit.php?post_type=job_listing', __('WPJM JobAdder Integration Settings', 'wpjm-jobadder'), __('JobAdder', 'wpjm-jobadder'), \JobAdder\Settings::get('capability'), 'wpjm-jobadder_settings', array(__CLASS__, 'page'));
        add_action('load-' . $page, array(__CLASS__, 'load_page'));
    }

    static function load_page() {
        if (!current_user_can(\JobAdder\Settings::get('capability'))) return;

        add_action('admin_enqueue_scripts', array(__CLASS__, 'admin_enqueue_scripts'));
    }

    static function admin_enqueue_scripts() {
        //wp_enqueue_script('wpjm-jobadder-admin', \JobAdder\Settings::get('dir') . 'assets/js/admin.js', array('jquery'), \JobAdder\Settings::get('version'), true);
    }

    static function register_settings() {
        register_setting('wpjm-jobadder', 'wpjm-jobadder_options'); //, array(__CLASS__, 'validate_options'));

        // API Settings

        add_settings_section('wpjm-jobadder_section_api', __('API Settings', 'wpjm-jobadder'), array(__CLASS__, 'section'), 'wpjm-jobadder');

        add_settings_field('wpjm-jobadder_client_id', __('Client Id', 'wpjm-jobadder'), array(__CLASS__, 'field_text'), 'wpjm-jobadder', 'wpjm-jobadder_section_api', array(
            'id' => 'settings_client_id',
        ));

        add_settings_field('wpjm-jobadder_client_secret', __('Client Secret', 'wpjm-jobadder'), array(__CLASS__, 'field_text'), 'wpjm-jobadder', 'wpjm-jobadder_section_api', array(
            'id' => 'settings_client_secret',
            'type' => 'password',
            'description' => sprintf(__('Register an application as a registered developer with JobAdder to get a client id and secret. Use this url, %s, as the redirect URI. See https://developers.jobadder.com/docs/ and contact api@jobadder.com for more details.', 'wpjm-jobadder'), self::get_url()),
        ));

        add_settings_field('wpjm-jobadder_authorized_ips', __('Authorized IP Addresses', 'wpjm-jobadder'), array(__CLASS__, 'field_text'), 'wpjm-jobadder', 'wpjm-jobadder_section_api', array(
            'id' => 'settings_authorized_ips',
            'description' => __('Separate IPv4 addresses by commas.', 'wpjm-jobadder'),
        ));

        add_settings_field('wpjm-jobadder_auth_code', __('Authorization Code', 'wpjm-jobadder'), array(__CLASS__, 'field_text'), 'wpjm-jobadder', 'wpjm-jobadder_section_api', array(
            'id' => 'settings_auth_code',
            'type' => 'password',
            'disabled' => true,
            'description' => __('For developer use only.', 'wpjm-jobadder'),
        ));

        add_settings_field('wpjm-jobadder_access_token', __('Access Token', 'wpjm-jobadder'), array(__CLASS__, 'field_text'), 'wpjm-jobadder', 'wpjm-jobadder_section_api', array(
            'id' => 'settings_access_token',
            'type' => 'password',
            'disabled' => true,
            'description' => __('For developer use only.', 'wpjm-jobadder'),
        ));

        add_settings_field('wpjm-jobadder_refresh_token', __('Refresh Token', 'wpjm-jobadder'), array(__CLASS__, 'field_text'), 'wpjm-jobadder', 'wpjm-jobadder_section_api', array(
            'id' => 'settings_refresh_token',
            'type' => 'password',
            'disabled' => true,
            'description' => __('For developer use only.', 'wpjm-jobadder'),
        ));

        add_settings_field('wpjm-jobadder_api_url', __('API Url', 'wpjm-jobadder'), array(__CLASS__, 'field_text'), 'wpjm-jobadder', 'wpjm-jobadder_section_api', array(
            'id' => 'settings_api_url',
            'type' => 'password',
            'disabled' => true,
            'description' => __('For developer use only.', 'wpjm-jobadder'),
        ));

        add_settings_field('wpjm-jobadder_webhook_id', __('WebHook Id', 'wpjm-jobadder'), array(__CLASS__, 'field_text'), 'wpjm-jobadder', 'wpjm-jobadder_section_api', array(
            'id' => 'settings_webhook_id',
            'type' => 'password',
            'disabled' => true,
            'description' => __('For developer use only.', 'wpjm-jobadder'),
        ));

        // Debug Settings

        add_settings_section('wpjm-jobadder_section_debug', __('Debug Settings', 'wpjm-jobadder'), array(__CLASS__, 'section'), 'wpjm-jobadder');

        add_settings_field('wpjm-jobadder_debug_enabled', __('Enabled', 'wpjm-jobadder'), array(__CLASS__, 'field_checkbox'), 'wpjm-jobadder', 'wpjm-jobadder_section_debug', array(
            'id' => 'settings_debug_enabled',
            'default' => \JobAdder\Settings::get_default('debug_enabled'),
        ));

        add_settings_field('wpjm-jobadder_debug_email', __('Email Address', 'wpjm-jobadder'), array(__CLASS__, 'field_text'), 'wpjm-jobadder', 'wpjm-jobadder_section_debug', array(
            'id' => 'settings_debug_email',
            'description' => __('Feed and webhook debugging information will be sent to this email as it comes in.', 'wpjm-jobadder'),
            'type' => 'email',
        ));
    }

    static function page() {
        if (!current_user_can(\JobAdder\Settings::get('capability'))) return;

        // Show error/update message
        settings_errors('wpjm-jobadder_messages');
?>
<div class="wrap">
    <h1><?php esc_html_e(get_admin_page_title()); ?></h1>
    <?php \JobAdder\API\Auth::settings(); ?>
    <form action="options.php" method="post"><?php
        settings_fields('wpjm-jobadder');
        do_settings_sections('wpjm-jobadder');
        submit_button(__('Save Settings', 'wpjm-jobadder'));
    ?></form>
</div>
<?php
    }

    static function validate_options($fields) {
        $valid_fields = array();

        foreach ($fields as $id => $field) {
            if (!isset($input[$id])) continue;
        }

        return apply_filters('validate_options', $valid_fields, $fields);
    }

    // Fields

    public static function section($args) {
        // <p>Description</p>
    }

    public static function field_text($args) {
        $value = '';
        $options = get_option('wpjm-jobadder_options');
        if (isset($args['default'])) $value = $args['default'];
        if (isset($args['id']) && !empty($args['id']) && isset($options[$args['id']]) && !empty($options[$args['id']])) $value = $options[$args['id']];

        $type = 'text';
        if (isset($args['type']) && !empty($args['type'])) {
            $type = $args['type'];
        }

        echo '<input class="regular-text" type="' . $type . '" value="' . esc_attr($value) . '"';

        if (isset($args['id']) && !empty($args['id'])) {
            echo ' id="' . esc_attr($args['id']) . '"';
            echo ' name="wpjm-jobadder_options[' . esc_attr($args['id']) . ']"';
        }
        if (isset($args['placeholder']) && !empty($args['placeholder'])) {
            echo ' placeholder="' . esc_attr($args['placeholder']) . '"';
        }
        if (isset($args['disabled']) && $args['disabled'] == true) {
            echo ' disabled="disabled"';
        }

        echo ' />';

        if (isset($args['description']) && !empty($args['description'])) {
            echo '<p class="description">';
            _e($args['description'], 'obfuscation');
            echo '</p>';
        }
    }

    public static function field_checkbox($args) {
        $value = false;
        $options = get_option('wpjm-jobadder_options');
        if (isset($args['default'])) $value = $args['default'];
        if (isset($args['id']) && !empty($args['id']) && isset($options[$args['id']]) && !empty($options[$args['id']])) $value = $options[$args['id']];

        echo '<input type="checkbox" value="1" type="checkbox" ' . checked($value, 1, false);

        if (isset($args['id']) && !empty($args['id'])) {
            echo ' id="' . esc_attr($args['id']) . '"';
            echo ' name="wpjm-jobadder_options[' . esc_attr($args['id']) . ']"';
        }
        echo ' />';

        if (isset($args['description']) && !empty($args['description'])) {
            echo '<p class="description">';
            esc_html_e($args['description'], 'wpjm-jobadder');
            echo '</p>';
        }
    }

    public static function get_url() {
        return admin_url('edit.php?post_type=job_listing&page=wpjm-jobadder_settings');
    }

}

\JobAdder\SettingsPage::init();
