<?php
/**
 * @package CleverOgre
 * @subpackage learndash-certificates-phpword
 * @version 0.1.0
 * @since 0.1.0
 */

namespace LearnDash\Certificate\PhpWord;

defined('ABSPATH') || exit;

class Display {
    use Singleton;

    public function __construct() {
        add_action('template_redirect', [$this, 'handle'], 4); // NOTE: learndash_certificate_display occurs at 5
    }

    public function validate():bool {
        if (!is_singular(learndash_get_post_type_slug('certificate'))) return false;
        if (!isset($_GET['cert-nonce']) || empty($_GET['cert-nonce'])) return false;
        return true;
    }

    public function handle() {
        // NOTE: Error handling is taken care of by learndash_certificate_display. Return early if invalid.
        if (!$this->validate()) return;

        $certificate_id = get_the_ID();
        $certificate_post = get_post($certificate_id);
        $view_user_id = get_current_user_id();

        $cert_user_id = get_current_user_id();
        if ((learndash_is_admin_user() || learndash_is_group_leader_user()) && isset($_GET['user']) && !empty($_GET['user'])) {
            $cert_user_id = absint($_GET['user']);
        }

        $cert_template_path = CertificateSettings::instance()->get_file_path($certificate_id);
        if (empty($cert_template_path)) return;

        $cert_type = false;
        $object_post_id = false;
        $cert_post_id = false;
        $allow_set_user = false;

        if (isset($_GET['group_id']) && !empty($_GET['group_id'])) {

            $group_id = absint($_GET['group_id']);
            if (!wp_verify_nonce(esc_attr($_GET['cert-nonce']), $group_id . $cert_user_id . $view_user_id)) return;

            $group_post = get_post($group_id);
            if (!$group_post || !is_a($group_post, 'WP_Post') || learndash_get_post_type_slug('group') !== $group_post->post_type) return;

            $group_certificate_post_id = learndash_get_setting($group_post->ID, 'certificate');
            if (absint($group_certificate_post_id) !== absint($certificate_post->ID)) return;

            $group_status = learndash_get_user_group_status($group_id, $cert_user_id, true);
            if ('completed' !== $group_status) return;

            $cert_type = 'group';
            $object_post_id = $group_id;
            $cert_post_id = absint($group_certificate_post_id);
            $allow_set_user = learndash_is_admin_user() || learndash_is_group_leader_user();

        } else if (isset($_GET['course_id']) && !empty($_GET['course_id'])) {

            $course_id = absint($_GET['course_id']);
            if (!wp_verify_nonce(esc_attr($_GET['cert-nonce']), $course_id . $cert_user_id . $view_user_id)) return;

            $course_post = get_post($course_id);
            if (!$course_post || !is_a($course_post, 'WP_Post') || learndash_get_post_type_slug('course') !== $course_post->post_type) return;

            $course_certificate_post_id = learndash_get_setting($course_post->ID, 'certificate');
            if (absint($course_certificate_post_id) !== absint($certificate_post->ID)) return;

            $course_status = learndash_course_status($course_id, $cert_user_id, true);
            if ('completed' !== $course_status) return;

            $cert_type = 'course';
            $object_post_id = $course_id;
            $cert_post_id = absint($course_certificate_post_id);
            $allow_set_user = learndash_is_admin_user() || learndash_is_group_leader_user();

        } elseif (isset($_GET['quiz']) && !empty($_GET['quiz'])) {

            $quiz_id = intval($_GET['quiz']);
            if (!wp_verify_nonce($_GET['cert-nonce'], $quiz_id . $cert_user_id . $view_user_id)) return;

            $quiz_post = get_post($quiz_id);
            if (!$quiz_post || !is_a($quiz_post, 'WP_Post') || learndash_get_post_type_slug('quiz') !== $quiz_post->post_type) return;

            $quiz_certificate_post_id = learndash_get_setting($quiz_post->ID, 'certificate');
            if (absint($quiz_certificate_post_id) !== absint($certificate_post->ID)) return;

            $time = isset($_GET['time']) ? intval($_GET['time']) : -1;
            $quizinfo = get_user_meta($cert_user_id, '_sfwd-quizzes', true);
            $selected_quizinfo = null;
            $selected_quizinfo2 = null;

            if (!empty($quizinfo)) {
                foreach ($quizinfo as $quiz_i) {
                    if (isset($quiz_i['time']) && intval($quiz_i['time']) == intval($time) && intval($quiz_i['quiz']) === intval($quiz_id)) {
                        $selected_quizinfo = $quiz_i;
                        break;
                    }
                    if (intval($quiz_i['quiz']) === intval($quiz_id)) {
                        $selected_quizinfo2 = $quiz_i;
                    }
                }
            }

            $selected_quizinfo = empty($selected_quizinfo) ? $selected_quizinfo2 : $selected_quizinfo;
            if (empty($selected_quizinfo)) return;

            $certificate_threshold = learndash_get_setting($selected_quizinfo['quiz'], 'threshold');

            if ((isset($selected_quizinfo['percentage']) && $selected_quizinfo['percentage'] >= $certificate_threshold * 100) || (isset($selected_quizinfo['count']) && $selected_quizinfo['score'] / $selected_quizinfo['count'] >= $certificate_threshold)) {
                $cert_type = 'quiz';
                $object_post_id = $quiz_id;
                $cert_post_id = absint($quiz_certificate_post_id);
                $allow_set_user = learndash_is_admin_user() || learndash_is_group_leader_user();
            }

        }

        if (!is_int($object_post_id) || !is_int($cert_post_id) || !$cert_type) return;

        if (intval($cert_user_id) !== intval($view_user_id)) {
            if (!$allow_set_user) return;
            wp_set_current_user($cert_user_id);
        }

        // TODO: Group & Quiz support
        $cert_doc_path = false;
        $cert_pdf_path = false;
        switch ($cert_type) {
            case 'course':
                $values = TemplateValues::instance()->get_all($cert_user_id, $object_post_id, true);

                $generator = TemplateGenerator::instance();
                $cert_doc_path = $generator->apply_template(
                    $cert_template_path,
                    $this->get_filename($object_post_id, $cert_post_id, $cert_user_id),
                    $values
                );
                if (!!$cert_doc_path) $cert_pdf_path = $generator->convert_pdf($cert_doc_path, true); // Will delete $cert_doc_path

                break;
        }
        if (!$cert_pdf_path) return;

        $this->output($cert_pdf_path, false, true); // Will delete $cert_pdf_path
    }

    private function get_object_title(int $object_id):string {
        return wp_strip_all_tags(get_the_title($object_id));
    }

    private function get_username(int $user_id, int $cert_id = 0) {
        $userdata = get_userdata($user_id);
        if (!is_a($userdata, '\WP_User')) return false;
        return apply_filters('learndash_pdf_username', $userdata->display_name, $user_id, $cert_id);
    }

    private function get_separator(bool $with_spaces = false) {
        $sep = apply_filters('document_title_separator', '-');
        return !!$with_spaces ? " {$sep} " : $sep;
    }

    private function get_title(int $object_id, int $cert_id, int $user_id) {
        $parts = array_filter([
            'username' => $this->get_username($user_id, $cert_id),
            'object_title' => $this->get_object_title($object_id),
            'cert_title' => $this->get_object_title($cert_id),
        ]);
        $parts = apply_filters('learndash_certificate_title_parts', $parts, $object_id, $cert_id, $user_id);
        if (empty($parts)) return false;
        return implode($this->get_separator(true), $parts);
    }

    private function get_filename(int $object_id, int $cert_id, int $user_id, string $type = 'title'):string {
        switch ($type) {
            case 'title':
                $filename = sanitize_file_name(str_replace($this->get_separator(true), $this->get_separator(false), $this->get_title($object_id, $cert_id, $user_id)));
                $filename = apply_filters('learndash_pdf_filename', $filename, $cert_id);
                break;
            default:
                $filename = strval($cert_id);
                break;
        }

        $filename = basename($filename);
        $filename = strtolower($filename);
        $filename = substr($filename, 0, 255);
        $filename = sanitize_file_name($filename);

        return $filename;
    }

    private function get_author_name(int $user_id, int $cert_id = 0) {
        return $this->get_username($user_id, $cert_id);
    }

    private function get_keywords(int $cert_id):string {
        $tags = wp_get_post_tags($cert_id);
        if (empty($tags)) return '';
        return implode(' ', wp_list_pluck($tags, 'name'));
    }

    private function output(string $path, bool $download = false, bool $delete = false) {
        $path_parts = pathinfo($path);

        header(sprintf(
            'Content-Type: %s',
            mime_content_type($path)
        ));
        header(sprintf(
            'Content-Length: %d',
            filesize($path)
        ));

        header('Cache-Control: private, must-revalidate, post-check=0, pre-check=0, max-age=1');
		header('Pragma: public');
		header('Expires: Sat, 26 Jul 1997 05:00:00 GMT');
		header('Last-Modified: '.gmdate('D, d M Y H:i:s').' GMT');

        header(sprintf(
            'Content-Disposition: %1$s; filename="%2$s"; filename*=UTF-8\'%2$s\'',
            (!!$download ? 'attachment' : 'inline'),
            rawurlencode($path_parts['filename'] . '.' . $path_parts['extension'])
        ));

        readfile($path);

        if (!!$delete && file_exists($path)) unlink($path);

        exit;
    }

}

Display::instance();
