<?php
/**
 * @package CleverOgre
 * @subpackage learndash-certificates-phpword
 * @version 0.1.1
 * @since 0.1.0
 */

namespace LearnDash\Certificate\PhpWord;

defined('ABSPATH') || exit;

use \Unoconv\Unoconv;
use \PhpOffice\PhpSpreadsheet\IOFactory as SpreadsheetIOFactory;
use \PhpOffice\PhpWord\IOFactory as WordIOFactory;
use \PhpOffice\PhpWord\TemplateProcessor;
use \PhpOffice\PhpWord\SimpleType\DocProtect;

class TemplateGenerator {
    use Singleton;

    private $unoconv = null;

    public function __construct() { }

    public static function get_allowed_mime_types() {
        return [
            // Word
            'doc' => 'application/msword',
            'docx' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'odt' => 'application/vnd.oasis.opendocument.text',
            'rtf' => 'application/rtf',

            // Spreadsheet
            'xls' => 'application/vnd.ms-excel',
            'xlsx' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'ods' => 'application/vnd.oasis.opendocument.spreadsheet',
            'csv' => 'text/csv',
        ];
    }

    private function get_vendor_path() {
        $path = Plugin::get_path() . 'vendor/autoload.php';
        if (!file_exists($path)) return false;
        return $path;
    }
    private function load_vendors():bool {
        $path = $this->get_vendor_path();
        if (!is_string($path) || empty($path)) return false;
        require_once($path);
        return true;
    }
    private function load_phpword():bool {
        if (class_exists('\PhpOffice\PhpWord\Settings')) return true;
        if (!$this->load_vendors()) return false;
        return class_exists('\PhpOffice\PhpWord\Settings');
    }
    private function load_phpspreadsheet():bool {
        if (class_exists('\PhpOffice\PhpSpreadsheet\IOFactory')) return true;
        if (!$this->load_vendors()) return false;
        return class_exists('\PhpOffice\PhpSpreadsheet\IOFactory');
    }

    public function apply_template(string $path, string $filename, array $values = []) {

        $path_parts = pathinfo($path);
        $tmp_path = sprintf('%s%s.%s', get_temp_dir(), $filename, $path_parts['extension']);

        switch ($path_parts['extension']) {
            case 'xls':
            case 'xlsx':
            case 'ods':
            case 'csv':
                if (!$this->load_phpspreadsheet()) return false;

                $spreadsheet = SpreadsheetIOFactory::load($path);
                $worksheet = $spreadsheet->getActiveSheet();

                foreach ($worksheet->getRowIterator() as $row) {
                    $cellIterator = $row->getCellIterator();
                    $cellIterator->setIterateOnlyExistingCells(true);
                    foreach ($cellIterator as $cell) {
                        $value = $cell->getValue();
                        if (strpos($value, '${') === false) continue;
                        if (preg_match_all('/\${([^}]+)}/', $value, $matches) === false) continue;
                        foreach ($matches[1] as $key) {
                            if (!isset($values[$key])) continue;
                            $value = str_replace('${'.$key.'}', $values[$key], $value);
                        }
                        $cell->setValue($value);
                    }
                }

                $writer = SpreadsheetIOFactory::createWriter($spreadsheet, ucfirst(strtolower(trim($path_parts['extension']))));
                $writer->save($tmp_path);
                break;
            case 'doc':
            case 'docx':
            case 'odt':
            case 'rtf':
                if (!$this->load_phpword()) return false;

                $templateProcessor = new TemplateProcessor($path);
                $templateProcessor->setValues($values);
                $templateProcessor->saveAs($tmp_path);
                break;
            default:
                return false;
        }

        return $tmp_path;

    }

    public function apply_password(string $path, string $password):string {
        if ((is_string($password) && empty($password)) || !is_string($password)) $password = self::generate_document_password();

        $path_parts = pathinfo($path);
        $tmp_path = sprintf('%s/%s_p.%s', sys_get_temp_dir(), $path_parts['filename'], $path_parts['extension']);

        if (!self::load_phpword()) return false;

        $phpWord = WordIOFactory::load($path);
        $documentProtection = $phpWord->getSettings()->getDocumentProtection();
        $documentProtection->setEditing(DocProtect::READ_ONLY);
        $documentProtection->setPassword($password);
        $phpWord->save($tmp_path);

        return $tmp_path;
    }

    private $converters = [
        '\LearnDash\Certificate\PhpWord\UnoconvConverter',
        '\LearnDash\Certificate\PhpWord\DOMPDFConverter',
    ];

    public function convert_pdf(string $path, bool $delete = true) {
        $this->load_vendors();

        foreach ($converters as $converter_class) {
            if (!class_exists($converter_class) || !$converter_class::valid()) continue;

            $converter = new $converter_class();
            if (!!($tmp_path = $convert->convert($path, '', $delete))) return $tmp_path;
        }

        return false;
    }

}
