<?php
/**
 * @package CleverOgre
 * @subpackage OgreSchema
 * @version 0.1.0
 * @since 0.1.0
 */

namespace OgreSchema;

if (!defined('ABSPATH')) exit;

class Data {

    public const SCHEMA_TYPES = ['business', 'person', 'product', 'event', 'organization', 'website', 'custom'];

    public static function get_schema():array {
        if (Settings::has_post_types() && is_singular(Settings::get_post_types())) {
            $schema_object = self::get_schema_object(get_the_ID());
            if (!empty($schema_object)) return $schema_object;
        }
        return self::get_schema_object();
    }

    public static function has_schema():bool {
        return !empty(self::get_schema());
    }

    public static function get_schema_object(string|int $id = 'option'):array {
        $cache_key = sprintf('ogreschema/get_schema_object_%s', strval($id));
        if (is_array($cache = wp_cache_get($cache_key))) return $cache;

        $schema_type = get_field('schema_type', $id);
        if (!in_array($schema_type, self::SCHEMA_TYPES)) return [];

        $acf_object = get_field('schema_' . $schema_type, $id);
        $schema_object = array();

        switch ($schema_type) {

            case 'business':
            case 'organization':
                if (isset($acf_object['type']) && !empty($acf_object['type']) && strtolower($acf_object['type']) != 'none') {
                    $schema_object['@type'] = $acf_object['type']; // type field with capitals but without spaces
                } else if ($schema_type == 'business') {
                    $schema_object['@type'] = 'LocalBusiness';
                } else if ($schema_type == 'organization') {
                    $schema_object['@type'] = 'Organization';
                } else {
                    return [];
                }

                if (isset($acf_object['name']) && !empty($acf_object['name'])) $schema_object['name'] = $acf_object['name'];

                if (isset($acf_object['urls']) && !empty($acf_object['urls']) && is_array($acf_object['urls'])) {
                    $schema_object['sameAs'] = array();
                    foreach ($acf_object['urls'] as $row) {
                        $schema_object['sameAs'][] = $row['url'];
                    }
                }

                if (isset($acf_object['logo']) && !empty($acf_object['logo']) && isset($acf_object['logo']['url']) && !empty($acf_object['logo']['url'])) $schema_object['logo'] = $acf_object['logo']['url'];
                if (isset($acf_object['image']) && !empty($acf_object['image']) && isset($acf_object['image']['url']) && !empty($acf_object['image']['url'])) $schema_object['image'] = $acf_object['image']['url'];

                if (isset($acf_object['description']) && !empty($acf_object['description'])) $schema_object['description'] = $acf_object['description'];

                if (isset($acf_object['address']) || isset($acf_object['city']) || isset($acf_object['state']) || isset($acf_object['postal_code']) || isset($acf_object['country'])) {
                    $schema_object['address'] = array('@type' => 'PostalAddress');
                    if (isset($acf_object['address']) && !empty($acf_object['address'])) $schema_object['address']['streetAddress'] = $acf_object['address'];
                    if (isset($acf_object['po_box']) && !empty($acf_object['po_box'])) $schema_object['address']['postOfficeBoxNumber'] = $acf_object['po_box'];
                    if (isset($acf_object['city']) && !empty($acf_object['city'])) $schema_object['address']['addressLocality'] = $acf_object['city'];
                    if (isset($acf_object['state']) && !empty($acf_object['state'])) $schema_object['address']['addressRegion'] = $acf_object['state'];
                    if (isset($acf_object['postal_code']) && !empty($acf_object['postal_code'])) $schema_object['address']['postalCode'] = $acf_object['postal_code'];
                    if (isset($acf_object['country']) && !empty($acf_object['country'])) $schema_object['address']['addressCountry'] = $acf_object['country'];
                }

                if (isset($acf_object['location']) && is_array($acf_object['location']) && !empty($acf_object['location'])) {
                    $schema_object['geo'] = array(
                        '@type' => 'GeoCoordinates',
                        'latitude' => $acf_object['location']['lat'],
                        'longitude' => $acf_object['location']['lng'],
                    );
                    $schema_object['hasMap'] = 'https://www.google.com/maps/place/' . urlencode($acf_object['location']['address']);
                }

                if (isset($acf_object['hours']) && !empty($acf_object['hours']) && is_array($acf_object['hours'])) {
                    $schema_object['openingHours'] = self::build_hours($acf_object['hours']);
                }

                if (isset($acf_object['price']) && is_numeric($acf_object['price']) && intval($acf_object['price']) >= 1 && intval($acf_object['price']) <= 5) {
                    $schema_object['priceRange'] = str_repeat('$', intval($acf_object['price']));
                }

                if (isset($acf_object['phone']) && !empty($acf_object['phone'])) {
                    $contact = array(
                        '@type' => 'ContactPoint',
                        'telephone' => $acf_object['phone'],
                    );
                    if (isset($acf_object['contact_type']) && !empty($acf_object['contact_type']) && strtolower($acf_object['contact_type']) != 'none') $contact['contactType'] = $acf_object['contact_type'];
                    if (isset($acf_object['contact_option']) && !empty($acf_object['contact_option']) && strtolower($acf_object['contact_option']) != 'none') $contact['contactOption'] = $acf_object['contact_option'];

                    $schema_object['contactPoint'] = $contact;
                    $schema_object['telephone'] = $acf_object['phone'];
                }

                break;

            case 'person':
                $schema_object['@type'] = 'person';

                if (isset($acf_object['name']) && !empty($acf_object['name'])) $schema_object['name'] = $acf_object['name'];
                if (isset($acf_object['job_title']) && !empty($acf_object['job_title'])) $schema_object['jobTitle'] = $acf_object['job_title'];

                if (isset($acf_object['address']) || isset($acf_object['po_box']) || isset($acf_object['city']) || isset($acf_object['state']) || isset($acf_object['postal_code']) || isset($acf_object['country'])) {
                    $schema_object['address'] = array('@type' => 'PostalAddress');
                    if (isset($acf_object['address']) && !empty($acf_object['address'])) $schema_object['address']['streetAddress'] = $acf_object['address'];
                    if (isset($acf_object['po_box']) && !empty($acf_object['po_box'])) $schema_object['address']['postOfficeBoxNumber'] = $acf_object['po_box'];
                    if (isset($acf_object['city']) && !empty($acf_object['city'])) $schema_object['address']['addressLocality'] = $acf_object['city'];
                    if (isset($acf_object['state']) && !empty($acf_object['state'])) $schema_object['address']['addressRegion'] = $acf_object['state'];
                    if (isset($acf_object['postal_code']) && !empty($acf_object['postal_code'])) $schema_object['address']['postalCode'] = $acf_object['postal_code'];
                    if (isset($acf_object['country']) && !empty($acf_object['country'])) $schema_object['address']['addressCountry'] = $acf_object['country'];
                }

                if (isset($acf_object['email']) && !empty($acf_object['email']) && is_email($acf_object['email'])) $schema_object['email'] = $acf_object['email'];
                if (isset($acf_object['phone']) && !empty($acf_object['phone'])) $schema_object['telephone'] = $acf_object['phone'];
                if (isset($acf_object['birth_date']) && !empty($acf_object['birth_date'])) $schema_object['birthDate'] = $acf_object['birth_date'];

                break;

            case 'product':
                $schema_object['@type'] = 'product';

                if (isset($acf_object['brand']) && !empty($acf_object['brand'])) $schema_object['brand'] = $acf_object['brand'];
                if (isset($acf_object['name']) && !empty($acf_object['name'])) $schema_object['name'] = $acf_object['name'];
                if (isset($acf_object['description']) && !empty($acf_object['description'])) $schema_object['description'] = $acf_object['description'];

                if (isset($acf_object['image']) && !empty($acf_object['image']) && isset($acf_object['image']['url']) && !empty($acf_object['image']['url'])) $schema_object['image'] = $acf_object['image']['url'];

                if (isset($acf_object['rating_value']) && !empty($acf_object['rating_value'])) {
                    $schema_object['aggregateRating'] = array(
                        '@type' => 'aggregateRating',
                        'ratingValue' => $acf_object['rating_value'],
                    );
                    if (isset($acf_object['review_count']) && $acf_object['review_count'] > 0) $schema_object['aggregateRating']['reviewCount'] = $acf_object['review_count'];
                }

                break;

            case 'event':
                if (isset($acf_object['type']) && !empty($acf_object['type']) && strtolower($acf_object['type']) != 'none') {
                    $schema_object['@type'] = $acf_object['type']; // type field with capitals but without spaces
                } else {
                    $schema_object['@type'] = 'Event';
                }

                if (isset($acf_object['name']) && !empty($acf_object['name'])) $schema_object['name'] = $acf_object['name'];
                if (isset($acf_object['description']) && !empty($acf_object['description'])) $schema_object['description'] = $acf_object['description'];
                if (isset($acf_object['date_start']) && !empty($acf_object['date_start'])) $schema_object['startDate'] = $acf_object['date_start'];
                if (isset($acf_object['date_end']) && !empty($acf_object['date_end'])) $schema_object['endDate'] = $acf_object['date_end'];

                if (isset($acf_object['venue_name']) || isset($acf_object['address'])) {
                    $location = array('@type' => 'Place');

                    if (isset($acf_object['venue_name']) && !empty($acf_object['venue_name'])) $location['name'] = $acf_object['venue_name'];
                    if (isset($acf_object['venue_url']) && !empty($acf_object['venue_url'])) $location['sameAs'] = $acf_object['venue_url'];

                    if (isset($acf_object['address']) && !empty($acf_object['address'])) {
                        $location['address'] = array(
                            '@type' => 'PostalAddress',
                            'streetAddress' => $acf_object['address'],
                        );

                        if (isset($acf_object['city']) && !empty($acf_object['city'])) $location['address']['addressLocality'] = $acf_object['city'];
                        if (isset($acf_object['state']) && !empty($acf_object['state'])) $location['address']['addressRegion'] = $acf_object['state'];
                        if (isset($acf_object['postal_code']) && !empty($acf_object['postal_code'])) $location['address']['postalCode'] = $acf_object['postal_code'];
                        if (isset($acf_object['country']) && !empty($acf_object['country'])) $location['address']['addressCountry'] = $acf_object['country'];
                    }

                    if (count($location) > 1) $schema_object['location'] = $location;
                }

                if (isset($acf_object['offer_description']) || isset($acf_object['offer_url']) || isset($acf_object['offer_price'])) {
                    $offer = array('@type' => 'Offer');

                    if (isset($acf_object['offer_description']) && !empty($acf_object['offer_description'])) $offer['description'] = $acf_object['offer_description'];
                    if (isset($acf_object['offer_url']) && !empty($acf_object['offer_url'])) $offer['url'] = $acf_object['offer_url'];
                    if (isset($acf_object['offer_price']) && !empty($acf_object['offer_price'])) $offer['price'] = $acf_object['offer_price'];

                    if (count($offer) > 1) $schema_object['offer'] = $offer;
                }

                break;

            case 'website':
                $schema_object['@type'] = 'WebSite';

                if (isset($acf_object['name']) && !empty($acf_object['name'])) $schema_object['name'] = $acf_object['name'];
                if (isset($acf_object['name_alternate']) && !empty($acf_object['name_alternate'])) $schema_object['alternateName'] = $acf_object['name_alternate'];

                break;

            case 'custom':
                if (isset($acf_object['fields']) && !empty($acf_object['fields'])) {
                    foreach ($acf_object['fields'] as $field) {
                        if (!isset($field['key']) || empty($field['key'])) continue;

                        switch ($field['type']) {
                            case 'array':
                                $schema_object[$field['key']] = array();
                                foreach ($field['array'] as $array_item) {
                                    if (!isset($array_item['value']) || empty($array_item['value'])) continue;
                                    $schema_object[$field['key']][] = $array_item['value'];
                                }
                                break;

                            case 'object':
                                $schema_object[$field['key']] = array();
                                foreach ($field['object'] as $object_item) {
                                    if (!isset($object_item['key']) || empty($object_item['key'])) continue;
                                    $schema_object[$field['key']][$object_item['key']] = $object_item['value'];
                                }
                                break;

                            default:
                                $schema_object[$field['key']] = $field['value'];
                                break;
                        }
                    }
                }

                break;

        }

        if (empty($schema_object)) return [];

        // Global Settings
        if (!isset($schema_object['url'])) $schema_object['url'] = get_site_url();
        if (!isset($schema_object['@context'])) $schema_object['@context'] = 'http://www.schema.org';
        
        $schema_object = apply_filters('ogreschema/get_schema_object', $schema_object, $id);

        // Set Cache
        wp_cache_set($cache_key, $schema_object, 'ogre_schema');
        return $schema_object;
    }

    private static function build_hours($hours_object) {
        $hours = '';
        foreach ($hours_object as $row) {
            if (!isset($row['day']) || empty($row['day'])) continue;
            if ($hours != '') $hours .= ', ';
            $hours .= $row['day'];
            if (isset($row['open']) && isset($row['close']) && !empty($row['open']) && !empty($row['close'])) {
                $hours .= ' ' . $row['open'] . '-' . $row['close'];
            }
        }
        return $hours;
    }

}
