<?php
/**
 * @package CleverOgre
 * @subpackage QLess
 * @since QLess 0.2.0
 */

if (!defined('ABSPATH')) exit;

if (!class_exists('QLessApi')) {
    class QLessApi {
        private $settings;

        private $cookie_authentication = false;
        private $cookie_identity = false;

        public function __construct() {
            $this->settings = new QLessSettings();
            $this->cookie_authentication = false;
            $this->cookie_identity = false;
        }

        // General API Functions

        public function wssid() {
            $xml = $this->request_xml(array('uri' => 'api/v1/wssid'));

            if ($xml != false) {
                return (string)$xml[0];
            } else {
                return false;
            }
        }

        public function kiosk_locations($location_ids) {
            if ($this->parse_ids($location_ids) == false) return false;

            $xml = $this->request_xml(array('uri' => "api/v1/kiosk/locations/{$location_ids}"));

            $data = array();
            foreach ($xml as $elem) {
                $location = array(
                    'id' => intval($elem->kioskInfo->id),
                    'title' => (string)$elem->description,
                );
                $data[] = $location;
            }

            return $data;
        }

        public function kiosk_queues($location_id, $queue_ids = false) {
            if ($this->parse_id($location_id) == false) return false;
            if ($queue_ids != false && $this->parse_ids($queue_ids) == false) return false;

            $xml = false;
            if ($queue_ids == false) {
                $xml = $this->request_xml(array('uri' => "api/v1/kiosk/location/{$location_id}/queues"));
            } else {
                $xml = $this->request_xml(array('uri' => "api/v1/kiosk/location/{$location_id}/queues/{$queue_ids}"));
            }

            $data = array();
            foreach ($xml as $elem) {
                // Get ID from child or as attribute
                if (isset($elem->kioskInfo) && isset($elem->kioskInfo->id)) {
                    $id = intval($elem->kioskInfo->id);
                } else if (isset($elem['id'])) {
                    $id = intval($elem['id']);
                } else {
                    continue;
                }

                $queue = array(
                    'id' => (int)$id,
                    'title' => (string)$elem->description,
                );
                $data[] = $queue;
            }

            return $data;
        }

        public function kiosk_waitInfo($queue_ids) {
            if ($this->parse_ids($queue_ids) == false) return false;

            $xml = $this->request_xml(array('uri' => "api/v1/kiosk/queues/{$queue_ids}/waitInfo"));

            $data = array();
            foreach ($xml as $elem) {
                $wait = array(
                    'queueId' => intval((string)$elem->attributes->queueId),
                    'peopleInLine' => intval((string)$elem->attributes->peopleInLine),
                    'description' => (string)$elem->forecastNextWaitTime,
                    'units' => is_object($elem->forecastNextWaitTime->attributes) ? (string)$elem->forecastNextWaitTime->attributes->units : '',
                    'duration' => is_object($elem->forecastNextWaitTime) ? (string)$elem->forecastNextWaitTime->attributes->duration : '',
                );
                $data[] = $wait;
            }

            return $data;
        }

        public function appointments($queue_ids, $query = false) {
            if ($this->parse_ids($queue_ids) == false) return false;

            $xml = false;
            if ($query !== false && is_string($query) && !empty($query)) {
                $xml = $this->request_xml(array(
                    'uri' => 'api/v1/employee/appointments/search',
                    'data' => array(
                        'queueIds' => $queue_ids,
                        'query' => $query,
                    ),
                    'cookies' => true,
                    //'authenticate' => true,
                    //'debug' => true,
                ));
            } else {
                $xml = $this->request_xml(array(
                    'uri' => 'api/v1/employee/appointments',
                    'data' => array(
                        'queueIds' => $queue_ids,
                    ),
                    'cookies' => true,
                    //'authenticate' => true,
                    //'wssid' => true,
                    //'debug' => true,
                ));
            }

            $data = array();
            foreach ($xml as $elem) {
                $appointment = array(
                    'status' => (string)$elem->attributes->status,
                    'start' => $this->parse_date((string)$elem->attributes->start),
                    'scheduled' => $this->parse_date((string)$elem->attributes->scheduled),
                    'queueId' => intval((string)$elem->attributes->queueId),
                    'customer' => array(
                        'id' => intval((string)$elem->customer->attributes->id),
                        'lastName' => (string)$elem->customer->consumer->attributes->lastName,
                        'firstName' => (string)$elem->customer->consumer->attributes->firstName,
                        'email' => (string)$elem->customer->consumer->attributes->email,
                        'phone' => (string)$elem->customer->consumer->phone,
                    ),
                    'serviceDuration' => (string)$elem->serviceDuration,
                );
                $data[] = $appointment;
            }

            return $data;
        }

        public $appointment_statuses = array('PENDING', 'APPROVED', 'REJECTED', 'CANCELLED');
        public function appointment_status($appointment_ids, $status) {
            if ($this->parse_ids($appointment_ids) == false || !in_array($status, $this->appointment_statuses)) return false;

            $xml = $this->request_xml(array(
                'uri' => "api/v1/employee/appointments/{$appointment_ids}/status",
                'post' => true,
                'data' => array(
                    'status' => $status,
                ),
                'authenticate' => true,
            ));

            return isset($xml->OK);
        }

        public function authenticate($echo = false) {
            if ($this->cookie_authentication !== false && !empty($this->cookie_authentication) && $this->cookie_identity !== false && !empty($cookie_identity)) return true;
            if (empty((string)$this->settings->api_userid) || empty((string)$this->settings->api_password)) return false;

            $cookies = $this->request_cookies(array(
                'uri' => 'authenticator',
                'data' => array(
                    'principal' => $this->settings->api_userid,
                    'credentials' => $this->settings->api_password,
                    'remember' => 'true',
                ),
                'post' => true,
                'echo' => $echo,
            ));

            if ($cookies == false || !is_array($cookies) || !isset($cookies['au']) || empty($cookies['au']) || !isset($cookies['i']) || empty($cookies['i'])) return false;

            $this->cookie_authentication = $cookies['au'];
            $this->cookie_identity = $cookies['i'];

            return array(
                'au' => $cookies['au'],
                'i' => $cookies['i'],
            );
        }

        public function current_user() {
            $xml = $this->request_xml(array(
                'uri' => 'api/v1/employee/employee',
                'post' => false,
                'cookies' => true,
                //'authenticate' => true,
                //'debug' => true,
            ));

            // Need to return data here
            return true;
        }

        // API Interface Functions

        protected function request($args) {
            $defaults = array(
                'uri' => '',
                'data' => array(),
                'post' => false,
                'authenticate' => false,
                'cookies' => false,
                'wssid' => false,
                'header' => false,
                'echo' => false,
                'debug' => false,
            );
            $args = wp_parse_args($args, $defaults);

            $ch = curl_init();

            $uri = $this->settings->api_uri . (substr($this->settings->api_uri, -1) === '/' ? '' : '/') . $args['uri'];
            if ($args['post'] == false && !empty($args['data'])) {
                $uri = add_query_arg($args['data'], $uri);
            }
            curl_setopt($ch, CURLOPT_URL, $uri);

            curl_setopt($ch, CURLOPT_FAILONERROR, 1);
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, 0);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($ch, CURLOPT_TIMEOUT, 15);

            if ($this->settings->ssl_enable == true) {
                //curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            }

            // Setup HTTP header if necessary
            $headers = array();
            if ($args['authenticate'] == true && $this->authenticate() != false) {
                $headers[] = "Cookie: au={$this->cookie_authentication}; Path=/";
                //$headers[] = "Cookie: i={$this->cookie_identity}; Path=/";
            }
            if ($args['wssid'] == true && ($wssid = $this->wssid()) != false) {
                $headers[] = 'X-WSSID-Authorization: ' . $wssid;
            }
            if (!empty($headers)) {
                curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
            }

            // Send client cookies
            if ($args['cookies'] == true && !empty($_COOKIE)) {
                $cookies = '';
                foreach ($_COOKIE as $key => $value) {
                    if (!empty($cookies)) $cookies .= ';';
                    $cookies .= $key . '=' . addslashes($value);
                }
                curl_setopt($ch, CURLOPT_COOKIE, $cookies);
                if ($args['debug'] == true) {
                    var_dump($_COOKIE);
                }
            }

            if ($args['post'] == true) {
                curl_setopt($ch, CURLOPT_POST, 1);
                if (!empty($args['data'])) {
                    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($args['data']));
                }
            }

            if ($args['header'] == true || $args['echo'] == true) {
                curl_setopt($ch, CURLOPT_HEADER, 1);
            }
            if ($args['echo'] == true) {
                curl_setopt($ch, CURLINFO_HEADER_OUT, 1);
            }

            if ($args['debug'] == true) {
                curl_setopt($ch, CURLOPT_VERBOSE, 1);
                $verbose = fopen('php://temp', 'w+');
                curl_setopt($ch, CURLOPT_STDERR, $verbose);
            }

            $response = curl_exec($ch);

            if ($args['echo'] == true) {
                // Header
                $header_size = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
                $header_text = substr($response, 0, $header_size);
                $header_arr = explode("\r\n", $header_text);
                foreach ($header_arr as $i => $line) {
                    list($key, $value) = explode(': ', $line);
                    switch ($key) {
                        // Replace
                        case 'Server':
                        case 'Date':
                        case 'Content-Length':
                        case 'Connection':
                        case 'Location':
                        case 'Strict-Transport-Security':
                            header($line, true);
                            break;
                        // Add
                        case 'Set-Cookie':
                            header($line, false);
                            break;
                    }
                }

                // Body Content
                echo substr($response, $header_size);
            }

            curl_close($ch);

            if ($args['debug'] == true) {
                echo '<div class="qless-debug qless-debug-api"><div class="qless-debug-inner">';
                if ($response === false && curl_errno($ch) != 0) {
                    printf("cUrl error (#%d): %s<br>\n", curl_errno($ch), esc_html(curl_error($ch)));
                }
                rewind($verbose);
                $verboseLog = stream_get_contents($verbose);
                echo "Verbose information:\n<pre>", esc_html($verboseLog), "</pre>\n";
                echo '</div></div>';
            }

            return $response;
        }

        protected function request_xml($args) {
            $response = $this->request($args);
            if ($response == false) return false;

            $xml = new SimpleXMLElement($response);
            return $xml;
        }

        protected function request_cookies($args) {
            if (!isset($args['header'])) $args['header'] = true;

            $response = $this->request($args);
            if ($response == false) return false;

            preg_match_all('/^Set-Cookie:\s*([^;]*)/mi', $response, $matches);
            $cookies = array();
            foreach ($matches[1] as $item) {
                parse_str($item, $cookie);
                $cookies = array_merge($cookies, $cookie);
            }

            return $cookies;
        }

        // General Functions

        public function parse_ids(&$ids) {
            if (!is_array($ids) && !is_string($ids) && !is_int($ids)) return false; // Check type

            // Check and parse string
            if (is_string($ids) && !(is_numeric($ids) || strpos($ids, ',') != false)) {
                return false;
            } else if (is_string($ids) && strpos($ids, ',') != false) {
                $ids = explode(',', $ids);
            }

            // Check and parse array
            if (is_array($ids) && empty($ids)) {
                return false;
            } else if (is_array($ids)) {
                $remove_ids = array();
                foreach ($ids as $key => $id) {
                    if (!is_string($id) || empty($id) || !is_numeric($id)) {
                        $remove_ids[] = $key;
                    } else {
                        $ids[$key] = trim($id);
                    }
                }

                if (!empty($remove_ids)) {
                    foreach ($remove_keys as $key) {
                        unset($ids[$key]);
                    }
                }

                if (empty($ids)) {
                    return false;
                } else {
                    $ids = implode(',', $ids);
                }
            } else {
                $ids = trim($ids);
            }

            // Check and parse int
            if (is_int($ids) && $ids <= 0) {
                return false;
            }

            return true;
        }

        private function parse_id(&$id) {
            if (!(is_string($id) || is_int($id)) || empty($id) || !is_numeric($id)) return false;
            if (is_string($id)) $id = trim($id);
            return true;
        }

        private function parse_date_object($date_string) {
            return DateTime::createFromFormat('c', $date_string);
        }

        private function parse_date($date_string) {
            return $this->parse_date_object($date_string)->format('Y-m-d H:i:s');
        }
    }

    function QLessApi() {
        global $QLessApi;
        if (!isset($QLessApi)) {
            $QLessApi = new QLessApi();
        }
        return $QLessApi;
    }

    QLessApi();

}
