<?php
/**
 * @package CleverOgre
 * @subpackage WP Job Manager - JobAdder Integration
 * @since WP Job Manager - JobAdder Integration 0.1.1
 */

namespace JobAdder\API;

if (!defined('ABSPATH')) exit;

class Auth {

    public static function init() {
        add_action('current_screen', array(__CLASS__, 'check_auth_code'));
    }

    static function check_auth_code() {
        if (!is_admin()) return;

        $current_screen = get_current_screen();
        if ($current_screen->id != 'job_listing_page_wpjm-jobadder_settings' || !isset($_GET['code'])) return;

        self::generate_access_token($_GET['code']);
        self::activate_webhook();
    }

    // URL used to retrieve authorization code
    public static function get_auth_url() {
        return \JobAdder\API\Utility::buildIdUrl(JOBADDER_ID_AUTHORIZE_ENDPOINT, '{api}{endpoint}', array(), array(
            'response_type' => 'code',
            'client_id' => \JobAdder\Settings::get('client_id'),
            'scope' => 'read write offline_access',
            'redirect_uri' => urlencode(\JobAdder\SettingsPage::get_url()),
        ));
    }

    // Return access token or refresh if needed
    public static function get_access_token() {
        $access_token = \JobAdder\Settings::get('access_token');

        // Make sure we have OAuth setup
        if (empty($access_token) || empty(\JobAdder\Settings::get('refresh_token'))) {
            return false;
        }

        // Check expire time
        if (intval(\JobAdder\Settings::get('token_expires')) - time() < 0) {
            $access_token = self::refresh_access_token();

            // Check if refresh failed
            if (!$access_token || empty($access_token)) return false;
        }

        return $access_token;
    }

    // Use auth code to get new access token
    public static function generate_access_token(string $auth_code) {
        if (empty($auth_code)) return false;

        // Store auth code for debug purposes
        \JobAdder\Settings::set('auth_code', $auth_code);

        // Request Access Token
        $data = \JobAdder\API\Utility::postForm(
            \JobAdder\API\Utility::buildIdUrl(JOBADDER_ID_TOKEN_ENDPOINT),
            array(
                'grant_type' => 'authorization_code',
                'code' => $auth_code,
                'redirect_uri' => \JobAdder\SettingsPage::get_url(),
                'client_id' => \JobAdder\Settings::get('client_id'),
                'client_secret' => \JobAdder\Settings::get('client_secret'),
            ),
            false
        );

        // Validate data and update tokens
        return self::update_token_data($data);
    }

    // Use refresh token to get new access token and refresh token
    public static function refresh_access_token($refresh_token = false) {
        if ($refresh_token == false) $refresh_token = \JobAdder\Settings::get('refresh_token');
        if (empty($refresh_token)) return false;

        // Request Access Token
        $data = \JobAdder\API\Utility::postForm(
            \JobAdder\API\Utility::buildIdUrl(JOBADDER_ID_TOKEN_ENDPOINT),
            array(
                'grant_type' => 'refresh_token',
                'client_id' => \JobAdder\Settings::get('client_id'),
                'client_secret' => \JobAdder\Settings::get('client_secret'),
                'refresh_token' => $refresh_token,
            ),
            false
        );

        // Validate data and update tokens
        return self::update_token_data($data);
    }

    private static function update_token_data($data) {
        if (!isset($data['access_token'])) return false;

        \JobAdder\Settings::set('access_token', (string)$data['access_token']);
        \JobAdder\Settings::set('token_expires', (int)(intval($data['expires_in']) + time()));
        \JobAdder\Settings::set('refresh_token', (string)$data['refresh_token']);
        \JobAdder\Settings::set('api_url', (string)$data['api']);

        return (string)$data['access_token'];
    }

    public static function activate_webhook() {
        // Remove old webhook if it exists
        self::delete_webhook();

        $access_token = \JobAdder\Settings::get('access_token');
        if (empty($access_token)) return false;

        // Post Webhook Activation Request
        $data = \JobAdder\API\Utility::postJson(
            \JobAdder\API\Utility::buildApiUrl(JOBADDER_WEBHOOK_ENDPOINT),
            array(
                'name' => __('Application Status', 'wpjm-jobadder'),
                'status' => 'Enabled',
                'url' => \JobAdder\Feed::get_webhook_url(),
                'events' => array(
                    'jobapplication_status_changed',
                ),
            )
        );
        if (!isset($data['webhookId'])) return false;

        // Update tokens
        \JobAdder\Settings::set('webhook_id', (string)$data['webhookId']);

        return (string)$data['webhookId'];
    }

    public static function delete_webhook($webhook_id = false) {
        if ($webhook_id == false) $webhook_id = \JobAdder\Settings::get('webhook_id');
        if (empty($webhook_id)) return false;

        // Perform DELETE
        if (!\JobAdder\API\Utility::delete(\JobAdder\API\Utility::buildApiIdUrl(JOBADDER_WEBHOOK_ENDPOINT, $webhook_id))) return false;

        // Clear plugin's webhook
        \JobAdder\Settings::set('webhook_id', '');

        return true;
    }

    // API Authorization section
    public static function settings() {
        if (empty(\JobAdder\Settings::get('client_id')) || empty(\JobAdder\Settings::get('client_secret'))) return;

        if (!isset($_GET['code'])) { ?>
<p id="wpjm-jobadder-auth">
    <a id="wpjm-jobadder-auth-link" class="button button-primary button-hero" href="<?php echo esc_url(self::get_auth_url()); ?>"><?php
        if (empty(\JobAdder\Settings::get('auth_code'))) {
            esc_html_e('Authorize with JobAdder', 'wpjm-jobadder');
        } else {
            esc_html_e('Reauthorize with JobAdder', 'wpjm-jobadder');
        }
    ?></a>
</p>
    <?php } else if (!empty(\JobAdder\Settings::get('access_token'))) { ?>
<div class="wpjm-jobadder-auth-notice notice notice-success is-dismissible">
    <p><?php esc_html_e('Successfully authenticated with JobAdder!', 'wpjm-jobadder'); ?></p>
    <button type="button" class="notice-dismiss">
        <span class="screen-reader-text"><?php esc_html_e('Dismiss this notice.', 'wpjm-jobadder'); ?></span>
    </button>
</div>
    <?php } else { ?>
<div class="wpjm-jobadder-auth-notice notice notice-error is-dismissible">
    <p><?php esc_html_e('Failed to authenticate with JobAdder.', 'wpjm-jobadder'); ?></p>
    <button type="button" class="notice-dismiss">
        <span class="screen-reader-text"><?php esc_html_e('Dismiss this notice.', 'wpjm-jobadder'); ?></span>
    </button>
</div>
    <?php }
    }

}

\JobAdder\API\Auth::init();
