<?php
/**
 * @package CleverOgre
 * @subpackage WP Job Manager - JobAdder Integration
 * @version 0.1.3
 * @since 0.1.1
 */

namespace JobAdder;

if (!defined('ABSPATH')) exit;

class Feed {

    public static function init() {
        add_action('rest_api_init', array(__CLASS__, 'api_init'));
        add_filter('query_vars', array(__CLASS__, 'register_query_var'), 10, 1);
        add_action('template_redirect', array(__CLASS__, 'process_query_feed'));
    }

    static function api_init() {
        // [site-url]/wp-json/wpjm-jobadder/v1/feed/
        register_rest_route('wpjm-jobadder/v1', '/feed/', array(
            'methods' => array('POST'),
            'callback' => array(__CLASS__, 'process_rest_feed'),
        ));

        // [site-url]/wp-json/wpjm-jobadder/v1/webhook/
        // For use with jobapplication_status_changed
        register_rest_route('wpjm-jobadder/v1', '/webhook/', array(
            'methods' => array('POST'),
            'callback' => array(__CLASS__, 'process_rest_webhook'),
        ));
    }

    public static function get_feed_url() {
        return get_site_url(null, '/wp-json/wpjm-jobadder/v1/feed/');
    }

    public static function get_webhook_url() {
        return get_site_url(null, '/wp-json/wpjm-jobadder/v1/webhook/');
    }

    static function register_query_var($query_vars) {
        $query_vars[] = self::get_query_var();
        return $query_vars;
    }

    static function process_rest_feed(\WP_REST_Request $request) {
        self::validate_request();
        self::process_feed($request->get_body());
    }

    static function process_rest_webhook(\WP_REST_Request $request) {
        self::validate_request();
        self::process_webhook($request->get_body());
    }

    static function process_query_feed() {
        if (!array_key_exists(self::get_query_var(), $GLOBALS['wp_query']->query_vars)) return;

        // Feed must be conducted on home url
        if ($_SERVER['REQUEST_URI'] != '/?' . self::get_query_var()) return;

        self::validate_request();
        self::process_feed(file_get_contents('php://input'));
    }

    private static function process_feed($xml_str) {
        try {
            // Parse XML
            $xml = simplexml_load_string(apply_filters('wpjm-jobadder_xml_feed', $xml_str), 'SimpleXMLElement', LIBXML_NOCDATA);
            if (empty($xml)) {
                wp_die(wpautop(__('The XML supplied was empty or invalid.', 'wpjm-jobadder')));
                exit;
            }

            // Email Debugging
            if (\JobAdder\Settings::get('debug_enabled') === '1' && is_email(\JobAdder\Settings::get('debug_email'))) {
                $xml_loc = wp_upload_dir()['basedir'] . '/jobadder-feed.xml';
                $xml->saveXML($xml_loc);

                wp_mail(\JobAdder\Settings::get('debug_email'), __('WPJM JobAdder Integration Job Listings Feed', 'wpjm-jobadder'), wpautop(__('The JobAdder job listings feed xml data is attached to this email.', 'wpjm-jobadder')), array(
                    'Content-Type: text/html; charset=UTF-8',
                ), array(
                    $xml_loc
                ));
            }

            // Parse feed jobs
            $updated_jobs = array();
            foreach ($xml->Job as $job) {
                $jid = (int)$job->attributes()->jid;

                if (\JobAdder\JobListing::update_job($job)) {
                    $updated_jobs[] = $jid;
                }
            }

            /* NOTE: Removed in case there are no active jobs
            if (empty($updated_jobs)) {
                wp_die(wpautop(__('Could not find any valid jobs in the provided XML.', 'wpjm-jobadder')));
                exit;
            } */

            // Update jobs list and trash old jobs
            \JobAdder\JobListing::update_jobs($updated_jobs);

            // Clear cache to force WP Job Manager to update
            if (class_exists('WP_Job_Manager_Cache_Helper')) {
                \WP_Job_Manager_Cache_Helper::get_transient_version('get_job_listings', true);
            }

            return __('Jobs updated successfully', 'wpjm-jobadder');
        } catch (Exception $e) {
            throw $e;
        }
    }

    private static function process_webhook($json_str) {
        try {
            $data = json_decode($json_str, true);
            if (!isset($data['event']) || !isset($data['job']['jobId'])) {
                wp_die(wpautop(__('JobAdder webhook json data improperly formatted or parsed.', 'wpjm-jobadder')));
                exit;
            }

            // Email Debugging
            if (\JobAdder\Settings::get('debug_enabled') === '1' && is_email(\JobAdder\Settings::get('debug_email'))) {
                $json_loc = wp_upload_dir()['basedir'] . '/jobadder-webhook.json';
                file_put_contents($json_loc, $json_str);

                wp_mail(\JobAdder\Settings::get('debug_email'), __('WPJM JobAdder Integration WebHook', 'wpjm-jobadder'), wpautop(__('The json data sent with this JobAdder WebHook is attached to this email.', 'wpjm-jobadder')), array(
                    'Content-Type: text/html; charset=UTF-8',
                ), array(
                    $json_loc
                ));
            }

            $job_id = (int)intval($data['job']['jobId']);
            if (!\JobAdder\API\Jobs::exists($jobId)) {
                wp_die(wpautop(sprintf(__('Unable to find job listing with job id, #%d.', 'wpjm-jobadder'), $job_id)));
                exit;
            }

            $processed = false;
            $processed = apply_filters(sprintf('wpjm-jobadder_webhook_%s', sanitize_title($data['event'])), $processed, $job_id, $data);
            $processed = apply_filters('wpjm-jobadder_webhook', $processed, $data['event'], $job_id, $data);

            if (!$processed) {
                wp_die(wpautop(sprintf(__('Webhook of type, %s, and for job id, #%d, unsuccessfully processed.', 'wpjm-jobadder'), $data['event'], $job_id)));
                exit;
            }

            return __('Webhook processed successfully', 'wpjm-jobadder');
        } catch (Exception $e) {
            throw $e;
        }
    }

    private static function validate_request() {
        // Make sure request is post not get
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            wp_die(wpautop(__('Invalid request method.', 'wpjm-jobadder')));
        }

        // Authorize IP address
        /* if (!self::is_authorized_ip()) {
            wp_die(wpautop(__('Invalid authorization.', 'wpjm-jobadder')));
        } */

        do_action('wpjm-jobadder_validate_request');
    }

    private static function get_authorized_ips() {
        $authorized_ips = \JobAdder\Settings::get('authorized_ips');
        $authorized_ips = explode(',', $authorized_ips);
        $authorized_ips = array_map('trim', $authorized_ips);

        return apply_filters('wpjm-jobadder_authorized_ips', $authorized_ips);
    }

    private static function is_authorized_ip() {
        $ips = self::get_authorized_ips();
        $auth = !empty($ips) ? in_array($_SERVER['REMOTE_ADDR'], $ips) : true;
        return apply_filters('wpjm-jobadder_request_authorized', $auth);
    }

    private static function get_query_var() {
        return apply_filters('wpjm-jobadder_query_var', 'jobadder');
    }

}

\JobAdder\Feed::init();
