<?php
/**
 * @package CleverOgre
 * @subpackage QLess
 * @since QLess 0.2.0
 */

if (!defined('ABSPATH')) exit;

if (!class_exists('QLessAjax')) {
    class QLessAjax {
        private $api;
        private $settings;

        public function __construct() {
            $this->api = QLessApi();
            $this->settings = new QLessSettings();

            add_action('wp_ajax_qless/build_error', array($this, 'build_error'));
            add_action('wp_ajax_qless/authenticate', array($this, 'authenticate'));
            add_action('wp_ajax_qless/get_appointments', array($this, 'get_appointments'));
            add_action('wp_ajax_qless/build_appointments', array($this, 'build_appointments'));
        }

        function build_error() {
            if (!isset($_POST['qless_nonce']) || !wp_verify_nonce($_POST['qless_nonce'], 'qless')) { // Check nonce
                echo $this::result_error(__('Invalid nonce field.', 'qless'));
            } else if (!isset($_POST['message']) || empty($_POST['message'])) { // Check error message
                echo $this::result_error(__('Invalid error message.', 'qless'));
            } else { // Print error
                echo $this::result_error($_POST['message']);
            }

            wp_die();
            exit;
        }

        function authenticate() {
            // Check nonce
            if (!isset($_POST['qless_nonce']) || !wp_verify_nonce($_POST['qless_nonce'], 'qless')) {
                //wp_send_json_error(__('Invalid nonce field.', 'qless'));
                wp_die();
                exit;
            }

            $cookies = $this->api->authenticate(true);
            /*
            if ($cookies != false) {
                wp_send_json_success($cookies);
            } else {
                wp_send_json_error(__('Failed to authenticate with QLess.', 'qless'));
            }
            */

            wp_die();
            exit;
        }

        function get_appointments() {
            // Check nonce
            if (!isset($_POST['qless_nonce']) || !wp_verify_nonce($_POST['qless_nonce'], 'qless')) {
                wp_send_json_error(__('Invalid nonce field.', 'qless'));
                wp_die();
                exit;
            }
            if (!isset($_POST['qless_form_nonce']) || !wp_verify_nonce($_POST['qless_form_nonce'], 'qless_appointments')) {
                wp_send_json_error(__('Invalid form nonce field.', 'qless'));
                wp_die();
                exit;
            }

            // Get Queue Ids
            $queue_ids = $_POST['qless_queue_ids'];
            if (!$queue_ids || $queue_ids === null || empty($queue_ids) || $this->api->parse_ids($queue_ids) == false) {
                wp_send_json_error(__('No queues set to be searched.', 'qless'));
                wp_die();
                exit;
            }

            // Set up query value
            $input_type = $_POST['qless_input'];
            if (!$input_type || $input_type === null || empty($input_type) || $input_type == 'none') {
                wp_send_json_error(__('No search type selected.', 'qless'));
                wp_die();
                exit;
            }

            $query = false;
            switch ($input_type) {
                case 'code':
                case 'phone':
                case 'email':
                    $query = $_POST['qless_' . $input_type];
                    break;
                case 'name':
                    $query = trim($_POST['qless_name_first'] . ' ' . $_POST['qless_name_last']);
                    break;
            }
            /* if ($query == false || empty($query)) {
                wp_send_json_error(__('Invalid search query.', 'qless'));
                wp_die();
                exit;
            } */

            // Perform API Request
            $appointments = $this->api->appointments($queue_ids, $query); //, $query);
            if ($appointments === false || !is_array($appointments)) {
                wp_send_json_error(__('QLess API request failure.', 'qless'));
                wp_die();
                exit;
            }

            wp_send_json_success($appointments);
            wp_die();
            exit;
        }

        function build_appointments() {
            // Check nonce
            if (!isset($_POST['qless_nonce']) || !wp_verify_nonce($_POST['qless_nonce'], 'qless')) {
                echo $this::result_error(__('Invalid nonce field.', 'qless'));
                wp_die();
                exit;
            }
            if (!isset($_POST['qless_form_nonce']) || !wp_verify_nonce($_POST['qless_form_nonce'], 'qless_appointments')) {
                echo $this::result_error(__('Invalid form nonce field.', 'qless'));
                wp_die();
                exit;
            }

            // Check if appointment data is set properly
            if (!isset($_POST['appointments'])) {
                echo $this::result_error(__('QLess API request failure.', 'qless'));
                wp_die();
                exit;
            }
            $appointments = json_decode($_POST['appointments']);
            if ($appointments === false || $appointments === null || empty($appointments)) {
                echo $this::result_none();
                wp_die();
                exit;
            }

            // Check if actions are enabled
            $actions = isset($_POST['qless_actions']) && (strtolower($_POST['qless_actions']) == 'true' || $_POST['qless_actions'] === true);

            // Generate appointment results
            foreach ($appointments as $data) {
                echo QLessAppointments::result_appointment($data, $actions);
            }

            wp_die();
            exit;
        }

        public static function result($table, $id, $status, $actions = false) {
            if (empty($table) || !is_string($id) || !is_string($status)) return false;

            // Opening tag
            $content = "<li class=\"qless-results-item\"><article id=\"qless-result-{$id}\" class=\"qless-result qless-result-status-{$status}\">";

            // Build Table
            $content .= '<table class="qless-result-table"><tbody>';
            foreach ($table as $id => $row) {
                $content .= "<tr class=\"{$id}\"><th class=\"label\">{$row['label']}</th><td class=\"value\">{$row['value']}</td></tr>";
            }
            $content .= '</tbody></table>';

            // Build Actions
            if ($actions !== false && !empty($actions)) {
                $content .= '<ul class="qless-result-actions">';

                foreach ($actions as $id => $action) {
                    $content .= "<li class=\"qless-result-action qless-result-action-{$id}\">";
                    $content .= "<a href=\"#\" title=\"{$action['title']}\">{$action['label']}</a>";
                    $content .= "</li>";
                }

                $content .= '</ul>';
            }

            // Closing Tag
            $content .= '</article></li>';

            return $content;
        }

        public static function result_initial() {
            return '<li class="qless-results-item qless-results-item-initial"><article id="qless-result-0" class="qless-result qless-result-initial"><p>' . __('Please search for an appointment in the form above.', 'qless') . '</p></article></li>';
        }

        public static function result_none() {
            return '<li class="qless-results-item qless-results-item-none"><article id="qless-result-0" class="qless-result qless-result-none"><p>' . __('No appointment results found.', 'qless') . '</p></article></li>';
        }

        public static function result_error($message) {
            return '<li class="qless-results-item qless-results-item-error"><article id="qless-result-o" class="qless-result qless-result-error"><p>' . $message . '</p></article></li>';
        }
    }

    function QLessAjax() {
        global $QLessAjax;
        if (!isset($QLessAjax)) {
            $QLessAjax = new QLessAjax();
        }
        return $QLessAjax;
    }

    QLessAjax();
}
