<?php
/**
 * @package CleverOgre
 * @subpackage QLess
 * @since QLess 0.2.0
 */

if (!defined('ABSPATH')) exit;

if (!class_exists('QLessAppointments')) {
    class QLessAppointments {
        private $api;
        private $settings;

        public function __construct() {
            $this->api = QLessApi();
            $this->settings = new QLessSettings();

            add_action('wp_enqueue_scripts', array($this, 'scripts'));
            //add_action('wp_ajax_qless/appointments', array($this, 'request'));
            //add_action('wp_ajax_qless/appointments_html', array($this, 'html'));
        }

        function scripts() {
            $min = defined('SCRIPT_DEBUG') && SCRIPT_DEBUG ? '' : '.min';

            wp_register_style('qless-appointments', $this->settings->dir . 'assets/css/qless.appointments.css', array(), $this->settings->version);

            wp_register_script('jquery-mask', $this->settings->dir . "assets/js/jquery.mask{$min}.js", array('jquery'), $this->settings->version, true);

            wp_register_script('qless-appointments', $this->settings->dir . "assets/js/qless.appointments{$min}.js", array('jquery', 'jquery-mask'), $this->settings->version, true);
            wp_localize_script('qless-appointments', 'qless', array(
                'ajaxurl' => admin_url('admin-ajax.php'),
                'api_uri' => $this->settings->api_uri,
                'ssl_enable' => $this->settings->ssl_enable,
                //'api_userid' => $this->settings->api_userid,
                //'api_password' => $this->settings->api_password,
            ));
        }

        function request() {
            // Check nonces
            if (!isset($_POST['qless_nonce']) || !wp_verify_nonce($_POST['qless_nonce'], 'qless')) {
                echo QLessAjax::result_error(__('Invalid nonce field.', 'qless'));
                wp_die();
                exit;
            }
            if (!isset($_POST['qless_form_nonce']) || !wp_verify_nonce($_POST['qless_form_nonce'], 'qless_appointments')) {
                echo QLessAjax::result_error(__('Invalid form nonce field.', 'qless'));
                wp_die();
                exit;
            }

            // Get Queue Ids
            $queue_ids = $_POST['qless_queue_ids'];
            if (!$queue_ids || $queue_ids === null || empty($queue_ids) || $this->api->parse_ids($queue_ids) == false) {
                echo QLessAjax::result_error(__('No queues set to be searched.', 'qless'));
                wp_die();
                exit;
            }

            // Set up query value
            $input_type = $_POST['qless_input'];
            if (!$input_type || $input_type === null || empty($input_type) || $input_type == 'none') {
                echo QLessAjax::result_error(__('No search type selected.', 'qless'));
                wp_die();
                exit;
            }

            $query = false;
            switch ($input_type) {
                case 'code':
                case 'phone':
                case 'email':
                    $query = $_POST['qless_' . $input_type];
                    break;
                case 'name':
                    $query = trim($_POST['qless_name_first'] . ' ' . $_POST['qless_name_last']);
                    break;
            }
            /* if ($query == false || empty($query)) {
                echo QLessAjax::result_error(__('Invalid search query.', 'qless'));
                wp_die();
                exit;
            } */

            // Perform API Request
            $appointments = $this->api->appointments($queue_ids, $query); //, $query);
            if ($appointments === false || !is_array($appointments)) {
                echo QLessAjax::result_error(__('QLess API request failure.', 'qless'));
                wp_die();
                exit;
            }

            // No results found
            if (empty($appointments)) {
                echo QLessAjax::result_none();
                wp_die();
                exit;
            }

            // Check if actions are enabled
            $actions = isset($_POST['qless_actions']) && (strtolower($_POST['qless_actions']) == 'true' || $_POST['qless_actions'] === true);

            // Generate appointment results
            foreach ($appointments as $data) {
                echo $this::result_appointment($data, $actions);
            }

            wp_die();
            exit;
        }

        function html() {

        }

        public function form($atts = null) {
            $defaults = array(
                'ids' => '', // Comma separated list of queue ids
                'code' => true,
                'name' => true,
                'phone' => true,
                'email' => true,
                'actions' => true,
            );
            $args = wp_parse_args($atts, $defaults);

            // Generate form html
            $html = '<div class="qless-appointments-container">';

            // Form
            $html .= '<form class="qless-form qless-form-appointments" method="post" enctype="multipart/form-data" action="#">';

            // Inputs
            $html .= '<div class="qless-form-body"><ul class="qless-fields">';

            // Search By Selector
            $inputs = array();
            if (!!$args['code']) $inputs['code'] = __('Confirmation Code', 'qless');
            if (!!$args['name']) $inputs['name'] = __('Name', 'qless');
            if (!!$args['phone']) $inputs['phone'] = __('Phone Number', 'qless');
            if (!!$args['email']) $inputs['email'] = __('Email Address', 'qless');
            if (count($inputs) > 1) {
                $html .= '<li class="qless-field qless-field-radio qless-field-input">';
                $html .= '<label class="qless-label" for="qless_input">' . __('Choose Search By', 'qless') . '</label>';
                $html .= '<div class="qless-input-container">';

                $html .= '<ul class="qless-radio">';
                $i = 0;
                foreach ($inputs as $id => $label) {
                    $input_id = 'qless_' . $id;
                    if ($id == 'name') $input_id = 'qless_name_first,qless_name_last';

                    $html .= '<li>';
                    $html .= "<input name=\"qless_input\" type=\"radio\" value=\"{$id}\" id=\"qless_input_{$i}\" data-input=\"{$input_id}\" " . checked($i, 0, false) . " />";
                    $html .= "<label for=\"qless_input_{$i}\">" . esc_html($label) . '</label>';
                    $html .= '</li>';
                    $i++;
                }
                $html .= '</ul>';

                $html .= '</div>';
                $html .= '</li>';
            }

            // Appointment confirmation code
            if (!!$args['code']) {
                $html .= '<li class="qless-field qless-field-code">';
                $html .= '<label class="qless-label" for="qless_code">' . __('Confirmation Code', 'qless') . '</label>';
                $html .= '<div class="qless-input-container"><input class="qless-input qless-input-code" name="qless_code" id="qless_code" type="text" value="" /></div>';
                $html .= '</li>';
            }

            // First and/or last name
            if (!!$args['name']) {
                $html .= '<li class="qless-field qless-field-name qless-field-name-first">';
                $html .= '<label class="qless-label" for="qless_name_first">' . __('First Name', 'qless') . '</label>';
                $html .= '<div class="qless-input-container"><input class="qless-input qless-input-name qless-input-name-first" name="qless_name_first" id="qless_name_first" type="text" value="" /></div>';
                $html .= '</li>';

                $html .= '<li class="qless-field qless-field-name qless-field-name-last">';
                $html .= '<label class="qless-label" for="qless_name_last">' . __('Last Name', 'qless') . '</label>';
                $html .= '<div class="qless-input-container"><input class="qless-input qless-input-name qless-input-name-last" name="qless_name_last" id="qless_name_last" type="text" value="" /></div>';
                $html .= '</li>';
            }

            // Last 4 digits of phone number
            if (!!$args['phone']) {
                $html .= '<li class="qless-field qless-field-phone">';
                $html .= '<label class="qless-label" for="qless_phone">' . __('Last 4 Digits of Phone Number', 'qless') . '</label>';
                $html .= '<div class="qless-input-container"><input class="qless-input qless-input-phone" name="qless_phone" id="qless_phone" type="text" value="" placeholder="' . __('####', 'qless') . '" data-mask="0000" data-mask-clearifnotmatch="true" data-mask-selectonfocus="true" /></div>';
                $html .= '</li>';
            }

            // Email address
            if (!!$args['email']) {
                $html .= '<li class="qless-field qless-field-email">';
                $html .= '<label class="qless-label" for="qless_email">' . __('Email Address', 'qless') . '</label>';
                $html .= '<div class="qless-input-container"><input class="qless-input qless-input-email" name="qless_email" id="qless_email" type="email" value="" placeholder="' . __('email@address.com', 'qless') . '" /></div>';
                $html .= '</li>';
            }

            $html .= '</ul></div>';

            // Footer
            $html .= '<div class="qless-form-footer">';
            $html .= '<button type="submit" class="qless-button">' . __('Search', 'qless') . '</button>';
            $html .= wp_nonce_field('qless', 'qless_nonce', true);
            $html .= wp_nonce_field('qless_appointments', 'qless_form_nonce', false);
            $html .= '<input type="hidden" name="qless_args" value="' . esc_attr(json_encode($args)) . '" />';
            if (!empty($args['ids'])) {
                $html .= '<input type="hidden" name="qless_queue_ids" value="' . esc_attr($args['ids']) . '" />';
            }
            if (count($inputs) == 0) {
                $html .= '<input type="hidden" name="qless_input" value="none" />';
            } else if (count($inputs) == 1) {
                $id = array_keys($inputs)[0];
                $input_id = 'qless_' . $id;
                if ($id == 'name') $input_id = 'qless_name_first,qless_name_last';

                $html .= "<input type=\"hidden\" name=\"qless_input\" value=\"{$id}\" data-input=\"{$input_id}\" />";
            }
            $html .= '<input type="hidden" name="qless_actions" value="' . (!!$args['actions'] ? 'true' : 'false') . '" />';
            $html .= '</div>';

            $html .= '</form>';

            // Results
            $html .= '<div class="qless-results-container"><ul class="qless-results">';
            $html .= QLessAjax::result_initial();
            $html .= '</ul></div>';

            $html .= '</div>';

            return $html;
        }

        public static function result_appointment($data, $actions = true) {
            // Format data
            if (isset($data['status']) && !empty($data['status'])) $data['status'] = strtolower($data['status']);

            // Build data table
            $table = array(
                'name' => array(
                    'label' => __('Name', 'qless'),
                    'value' => trim("{$data['customer']['firstName']} {$data['customer']['lastName']}"),
                ),
            );
            if (!empty($data['customer']['email']) && is_email($data['customer']['email'])) {
                $table['email'] = array(
                    'label' => __('Email Address', 'qless'),
                    'value' => "<a href=\"mailto:{$data['customer']['email']}\" target=\"_blank\">{$data['customer']['email']}</a>",
                );
            }
            if (!empty($data['customer']['phone'])) {
                $number = preg_replace('/[^0-9]/', '', $data['customer']['phone']);
                $table['phone'] = array(
                    'label' => __('Phone Number', 'qless'),
                    'value' => "<a href=\"tel:{$number}\" target=\"_blank\">{$data['customer']['phone']}</a>",
                );
            }
            if (!empty($data['start'])) {
                $table['start'] = array(
                    'label' => __('Start', 'qless'),
                    'value' => $data['start'],
                );
            }
            if (!empty($data['serviceDuration'])) {
                $table['duration'] = array(
                    'label' => __('Duration', 'qless'),
                    'value' => $data['serviceDuration'],
                );
            }
            if (!empty($data['scheduled'])) {
                $table['scheduled'] = array(
                    'label' => __('Scheduled', 'qless'),
                    'value' => $data['scheduled'],
                );
            }
            if (!empty($data['status'])) {
                $table['status'] = array(
                    'label' => __('Status', 'qless'),
                    'value' => ucfirst($data['status']),
                );
            }

            $actions_arr = array();
            if ($actions == true && !in_array($data['status'], array('rejected', 'cancelled'))) {
                switch ($data['status']) {
                    case 'pending':
                        $actions_arr['approve'] = array(
                            'title' => __('Set as Approved', 'qless'),
                            'label' => __('Approve', 'qless'),
                        );
                        $actions_arr['reject'] = array(
                            'title' => __('Set as Rejected', 'qless'),
                            'label' => __('Reject', 'qless'),
                        );
                        break;

                    case 'approved':
                        $actions_arr['cancel'] = array(
                            'title' => __('Cancel this Appointment', 'qless'),
                            'label' => __('Cancel', 'qless'),
                        );
                        break;
                }
            }

            return QLessAjax::result($table, $data['queueId'], $data['status'], $actions_arr);
        }
    }

    function QLessAppointments() {
        global $QLessAppointments;
        if (!isset($QLessAppointments)) {
            $QLessAppointments = new QLessAppointments();
        }
        return $QLessAppointments;
    }

    QLessAppointments();
}
