<?php
/**
 * @package CleverOgre
 * @subpackage OgreSchema
 * @version 0.1.0
 * @since 0.1.0
 */

namespace OgreSchema;

defined('ABSPATH') || exit;

abstract class PluginBase {
    use Singleton;

    const REQUIREMENT_NONE = 0x01;
    const REQUIREMENT_PLUGIN = 0x02;
    const REQUIREMENT_CLASS = 0x04;

    protected string $file;
    protected array $requirements;
    protected array $files;
    private array $plugin_data;

    public function __construct(string $file = __FILE__) {
        $this->file = $file;
        $this->requirements = [];
        $this->files = [];
        $this->plugin_data = [];

        // Load composer packages
        $this->setup_composer();

        // Setup Plugin Updates
        $this->setup_updater();

        // Set Text Domain
        add_action('init', [$this, 'setup_textdomain']);

        // Requirements Check
        add_action('after_setup_theme', [$this, 'check_requirements'], 1, 0);

        $this->enable();
    }

    // Plugin Initialization

    protected function enable() {
        add_action('after_setup_theme', [$this, 'load'], 10, 0);
        add_filter('acf/settings/load_json', [$this, 'register_acf_json'], 10, 1);
    }
    protected function disable() {
        remove_action('after_setup_theme', [$this, 'load'], 10, 0);
        remove_filter('acf/settings/load_json', [$this, 'register_acf_json'], 10, 1);
    }

    public function load() {
        $this->load_files();
    }

    public function register_acf_json(array $paths):array {
        $path = $this->get_path() . 'acf-json';
        if (!in_array($path, $paths)) $paths[] = $path;
        return $paths;
    }

    // Setup

    protected function setup_composer() {
        require_once($this->get_path() . 'vendor/autoload.php');
    }

    protected function setup_updater() {
        global $wppul_plugin_file, $wppul_server, $wppul_license_required;
        $wppul_plugin_file = $this->get_file();
        $wppul_server = 'https://plugins.cleverogre.com';
        $wppul_license_required = false;
        if (!class_exists('Puc_v4_Factory')) require_once($this->get_path() . 'lib/wp-package-updater-lib/plugin-update-checker/plugin-update-checker.php');
        require_once($this->get_path() . 'lib/wp-package-updater-lib/package-updater.php');
    }

    public function setup_textdomain() {
        load_plugin_textdomain($this->get_textdomain(), false, plugin_basename(dirname($this->file)) . '/lang');
    }

    // Requirements

    protected function add_requirement(string $key, array $data) {
        $this->requirements[$key] = wp_parse_args($data, [
            'name' => '',
            'url' => '',
            'type' => self::REQUIREMENT_NONE,
            'plugin' => '',
            'class' => '',
        ]);
    }

    public function check_requirements() {
        $invalid = false;
        foreach ($this->requirements as $key => $data) {
            if ($data['type'] & self::REQUIREMENT_NONE) continue;
            if ($data['type'] & self::REQUIREMENT_PLUGIN && !empty($data['plugin']) && function_exists('is_plugin_active') && is_plugin_active($data['plugin'])) continue;
            if ($data['type'] & self::REQUIREMENT_CLASS && !empty($data['class']) && class_exists($data['class'])) continue;

            $this->add_requirement_notice($data);
            $invalid = true;
        }

        if (!!$invalid) $this->disable();
    }

    protected function add_requirement_notice($data) {
        add_action('admin_notices', function () use ($data) {
            $message = sprintf(
                __('In order to use the %1$s plugin, it is required that you install and activate the %3$s plugin. You can do this on the <a href="%2$s">plugins</a> page when logged in as an administrator. To download this plugin, visit the <a href="%4$s" target="_blank">%3$s website</a>.', $this->get_textdomain()),
                $this->get_title(),
                esc_url(admin_url('plugins.php')),
                esc_html($data['name']),
                esc_url($data['url'])
            );
            printf('<div class="%s"><p>%s</p></div>', esc_attr('notice notice-error'), wpautop(wp_kses_post($message)));
        });
    }

    // Plugin Files

    protected function add_file(string $relpath):bool {
        if (!is_string($relpath) || empty($relpath)) return false;
        $this->files[] = $relpath;
        return true;
    }
    protected function add_files(array $relpaths):bool {
        $valid = true;
        foreach ($relpaths as $relpath) {
            if (!$this->add_file($relpath)) $valid = false;
        }
        return $valid;
    }

    protected function load_files():bool {
        if (empty($this->files)) return false;
        $valid = true;
        foreach ($this->files as $relpath) {
            $path = rtrim($this->get_path(), '/') . '/' . ltrim($relpath, '/');
            if (!file_exists($path)) $valid = false;
            else include_once($path);
        }
        return $valid;
    }

    // Plugin Data Accessors

    private function get_data(string $key):string {
        if (!function_exists('get_plugin_data')) require_once(ABSPATH . 'wp-admin/includes/plugin.php');
        if (!is_array($this->plugin_data) || empty($this->plugin_data)) {
            $this->plugin_data = get_plugin_data($this->file, translate: false);
        }
        if (!array_key_exists($key, $this->plugin_data)) return '';
        return $this->plugin_data[$key];
    }

    public function get_textdomain():string {
        return $this->get_data('TextDomain');
    }

    public function get_id():string {
        return $this->get_textdomain();
    }

    public function get_version():string {
        return $this->get_data('Version');
    }

    public function get_title():string {
        return __($this->get_data('Name'), $this->get_textdomain());
    }

    public function get_description():string {
        return __($this->get_data('Description'), $this->get_textdomain());
    }

    // Plugin File Path Calculations

    public function get_file():string {
        return $this->file;
    }

    public function get_path(string $file = ''):string {
        if (empty($file)) $file = $this->file;
        return trailingslashit(dirname($file));
    }

    public function get_dir(string $file = ''):string {
        $dir = $this->get_path($file);
        $count = 0;

        // Sanitize for Win32 installs
        $dir = str_replace('\\', '/', $dir);

        // If file is in plugins folder
        $wp_plugin_dir = str_replace('\\', '/', WP_PLUGIN_DIR);
        $dir = str_replace($wp_plugin_dir, plugins_url(), $dir, $count);

        if ($count < 1) {
            // If file is in wp-content folder
            $wp_content_dir = str_replace('\\', '/', WP_CONTENT_DIR);
            $dir = str_replace($wp_content_dir, content_url(), $dir, $count);
        }

        if ($count < 1) {
            // If file is in ??? folder
            $wp_dir = str_replace('\\', '/', ABSPATH);
            $dir = str_replace($wp_dir, site_url('/'), $dir);
        }

        return $dir;
    }

    public function get_hook(string $file = ''):string {
        if (empty($file)) $file = $this->file;
        return basename(dirname($file)) . '/' . basename($file);
    }

    public function get_url(string $file = ''):string {
        if (empty($file)) $file = $this->file;
        return plugin_dir_url($file);
    }

    // Static Translation Functions

    public static function __(string $text):string {
        return __($text, self::instance()->get_textdomain());
    }

    public static function esc_html__(string $text):string {
        return esc_html__($text, self::instance()->get_textdomain());
    }

    public static function esc_attr__(string $text):string {
        return esc_attr__($text, self::instance()->get_textdomain());
    }

}
