<?php
/**
 * @package CleverOgre
 * @subpackage QLess
 * @since QLess 0.2.0
 */

if (!defined('ABSPATH')) exit;

if (!class_exists('QLessSettings') && !class_exists('QLessSettingsPage')) {

    class QLessSettings {

        private $defaults; // Value set in get function

        // Plugin Settings
        public $name;
        public $version;
        public $capability;
        public $path;
        public $dir;
        public $hook;

        // Admin Settings
        public $license_key;

        public $api_uri;
        public $ssl_enable;
        public $api_userid;
        public $api_password;

        public function __construct() {
            // Use getter/setter if not set
            foreach ($this as $key => $value) {
                unset($this->$key);
            }

            $options = get_option('qless_options');
            foreach ($this->defaults as $key => $val) {
                if (isset($options['qless_settings_' . $key])) continue;
                $this->$key = $val;
            }
        }

        public function __set($key, $value) {
            if (property_exists($this, $key)) {
                switch ($key) {
                    /* // Read-only properties
                    case 'readonly':
                        break;
                    */

                    default:
                        $options = get_option('qless_options');
                        $options['qless_settings_' . $key] = $value;
                        update_option('qless_options', $options);
                        break;
                }
            }
            return $this;
        }

        public function __get($key) {
            if (property_exists($this, $key)) {
                switch ($key) {
                    // Constant properties
                    case 'defaults':
                        return array(
                            'license_key' => '',
                            'api_uri' => 'https://merchant.qless.com/qless/api/v1',
                            'api_userid' => 'username',
                            'api_password' => 'password',
                            'ssl_enable' => true,
                        );
                        break;

                    default:
                        $options = get_option('qless_options');

                        if (empty($options['qless_settings_' . $key])) {
                            return $this->defaults[$key];
                        } else {
                            return $options['qless_settings_' . $key];
                        }
                        break;
                }
            }
            return null;
        }
    }

    class QLessSettingsPage {
        private $settings;

        public function __construct() {
            $this->settings = new QLessSettings();

            add_action('admin_init', array($this, 'init'));
            add_action('admin_menu', array($this, 'menu'));
        }

        function init() {
            register_setting('qless', 'qless_options');

            // Plugin Settings

            add_settings_section('qless_section_settings', __('Plugin Settings', 'qless'), array($this, 'section'), 'qless');

            add_settings_field('qless_license_key', __('License Key', 'qless'), array($this, 'field_text'), 'qless', 'qless_section_settings', array(
                'id' => 'qless_settings_license_key',
                'description' => 'Contact <a href="http://www.cleverogre.com/" target="_blank">CleverOgre</a> for a license key for this product. Updates will not work without a valid license key.',
            ));

            add_settings_section('qless_section_api', __('API Settings', 'qless'), array($this, 'section'), 'qless');

            add_settings_field('qless_api_uri', __('API Uri', 'qless'), array($this, 'field_text'), 'qless', 'qless_section_api', array(
                'id' => 'qless_settings_api_uri',
                'description' => 'Visit <a href="https://classic2.qless.com/partner/webServices/" target="_blank">QLess</a> for more information.',
            ));

            add_settings_field('qless_ssl_enable', __('Enable SSL?', 'qless'), array($this, 'field_checkbox'), 'qless', 'qless_section_api', array(
                'id' => 'qless_settings_ssl_enable',
            ));

            add_settings_field('qless_api_userid', __('Employee Userid', 'qless'), array($this, 'field_text'), 'qless', 'qless_section_api', array(
                'id' => 'qless_settings_api_userid',
            ));

            add_settings_field('qless_api_password', __('Employee Password', 'qless'), array($this, 'field_text'), 'qless', 'qless_section_api', array(
                'id' => 'qless_settings_api_password',
                'type' => 'password'
            ));
        }

        function menu() {
            if (current_user_can($this->settings->capability)) {
                add_options_page(__('QLess Settings', 'qless'), __('QLess', 'qless'), $this->settings->capability, 'qless_settings', array($this, 'page'));
            }
        }

        function page() {
            if (!current_user_can($this->settings->capability)) {
                return;
            }

            // Show error/update messages
            settings_errors('qless_messages');

            ?>
            <div class="wrap">
                <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
                <form action="options.php" method="post"><?php
                    settings_fields('qless');
                    do_settings_sections('qless');
                    submit_button(__('Save Settings', 'qless'));
                ?></form>
            </div>
            <?php
        }

        // Sections

        function section($args) { }

        // Fields

        function field_text($args) {
            $options = get_option('qless_options');

            $type = 'text';
            if (isset($args['type']) && !empty($args['type'])) {
                $type = $args['type'];
            }

            echo '<input class="regular-text" type="' . $type . '"';

            if (isset($args['id']) && !empty($args['id'])) {
                echo ' id="' . esc_attr($args['id']) . '"';
                echo ' name="qless_options[' . esc_attr($args['id']) . ']"';
                echo ' value="' . esc_attr($options[$args['id']]) . '"';
            }
            if (isset($args['placeholder']) && !empty($args['placeholder'])) {
                echo ' placeholder="' . esc_attr($args['placeholder']) . '"';
            }

            echo ' />';

            if (isset($args['description']) && !empty($args['description'])) {
                echo '<p class="description">';
                //esc_html_e($args['description'], 'qless');
                _e($args['description'], 'qless');
                echo '</p>';
            }
        }

        function field_checkbox($args) {
            $options = get_option('qless_options');

            echo '<input type="checkbox" value="1" type="checkbox"';

            if (isset($args['id']) && !empty($args['id'])) {
                echo ' id="' . esc_attr($args['id']) . '"';
                echo ' name="qless_options[' . esc_attr($args['id']) . ']"';
                echo ' ' . checked($options[$args['id']], 1, false);

            }
            echo ' />';

            if (isset($args['description']) && !empty($args['description'])) {
                echo '<p class="description">';
                esc_html_e($args['description'], 'qless');
                echo '</p>';
            }
        }

        function field_wysiwyg($args) {
            if (!isset($args['id']) || empty($args['id'])) {
                return;
            }

            $options = get_option('qless_options');

            $settings = array(
                'tinymce' => true,
                'textarea_name' => 'qless_options[' . $args['id'] . ']',
                'textarea_rows' => 15,
                'tabindex' => 1,
            );
            wp_editor($options[$args['id']], $args['id'], $settings);

            if (isset($args['description']) && !empty($args['description'])) {
                echo '<p class="description">';
                esc_html_e($args['description'], 'qless');
                echo '</p>';
            }
        }

        function field_select($args) {
            if (!isset($args['id']) || empty($args['id'])) return;
            $options = get_option('qless_options');

            echo '<select id="' . esc_attr($args['id']) . '" name="qless_options[' . esc_attr($args['id']) . ']">';

            if (isset($args['placeholder']) && !empty($args['placeholder'])) {
                echo '<option value="">' . esc_html($args['placeholder']) . '</option>';
            }

            $selected = $options[$args['id']];
            if (empty($selected) && isset($args['default']) && !empty($args['default'])) {
                $selected = $args['default'];
            }

            foreach ($args['options'] as $value => $title) {
                echo '<option value="' . esc_attr($value) . '" ' . selected($selected, $value, false) . '>' . esc_html($title) . '</option>';
            }

            echo '</select>';

            if (isset($args['description']) && !empty($args['description'])) {
                echo '<p class="description">';
                //esc_html_e($args['description'], 'qless');
                _e($args['description'], 'qless');
                echo '</p>';
            }
        }
    }

    if (is_admin()) {
        new QLessSettingsPage();
    }
}
